#Author: Joseph Haraldson
#Copyright: 2015, Joseph Haraldson
#

###########
#BUG FIXES#
#There is a problem in ComputeGCRDVector
# where we compute a nullspace vector for V instead of blowing away
#high order terms, we blow away all the terms. The fix is to iterate through
#different null space vectors. It's deterministic and will terminate and find 
#a solution, we just don't need to do it right now...
#
#
#
#Small bug with switching mult for GCLD computations...
###########


#proc  DPhi
#Input: Ore polynomis f, the dimension of the vector and underlying algebric ring R
#Output:  deg row vector representation for f, padded with sufficiently many zeros
#This is the standard form in the paper that we have discussed.
#to obtain Dphi(g) just switch f and g as arguments, if it is desired
DPhi:=proc(f, deg, R)
  local n, m, i, H, j, vec; 
  n := degree(f, R[1]);
#  m := degree(g, R[1]); 
  vec := CoefficientVector(f, R[1]);
  H := Vector[row](deg); 
  for i from 1 to n+1 do
    H(i) := vec(i)
  end do;
return H;
 end proc;

#############################################################
#############################################################
#############################################################


#proc DifferentialVandermonde
#Input: Ore polynomials f,g and underlying algebreic ring R
#Output: Differential block Vandermonde matrix representation of f and g. 
#Used in null space computation to determine the exact GCRD of f and g. 

DifferentialVandermonde:= proc(f,g,R)
#delcare some   local here in a bit
local i,j, m,n, Fphi, Gphi, DOP,V; # maybe this is sufficient. 

DOP:=R[1]; #differential operator. it should be Dt, Dz or Dx or something like that 
m := degree(f,R[1]);
n := degree(g,R[1]); 

#make the highest degree one first. This is a hack to get around a bug.
#if(n>m) then return DifferentialVandermonde(g,f,R) fi;

V := Matrix(m+n,m+n); #create initially empty (m+n) x (m+n) matrix 

#what out for the dimensions. If f has degree n then we go from 0-> n-1 <=> 1  -> n
#fill up the f portion of the Vandermonde matrix
for i from 1 to n do  
  V[i]:=  DPhi(mult(DOP^(i-1),f,R),n+m,R);
  #we do DOP^(i-1) because we are correcting an "off by 1" since the polynomials are index from 0 to n-1 (and we index from 1 to n);
od;
#fill up the g portion of the Vandermonde matrix
for i from n+1 to m+n do
  #we do DOP^(i-1) because we are correcting an "off by 1" since the polynomials are index from 0 to n-1 (and we index from 1 to n);
  V[i]:= DPhi(mult(DOP^(i-1-n),g,R),n+m,R); 
od;
V;#let us hope V is singular and what not shall we have.
end proc;

#############################################################
#############################################################
#############################################################

#Proc: DifferentialSubResultant
#Input: Ore polynomials f,g of degree $m$ and $n$ respectivel
#	and indices i,j of which sub-resultant we want. 
#Output: (m+n-i) X (m+n-j) subresultant matrix
#Purpose: Used for setting up some systems of equations. the 
#	 name might need to be changed a bit since this isn't
#	a proper subresultant in the sense of Ziming Li.

DifferentialSubResultant:=proc(f,g,R,i,j)
local m,n,S,ix,N;
m:=degree(f,R[1]);
n:=degree(g,R[1]);

#TODO: put a sanity check here about $i$ and $j$.

#double check the dims. I might have an off by 1.
#N:=max(m+n-i+1,m+n-j+1);
N:=m+n-i-j+1;
S:=Matrix(m+n-i-j+2,m+n-j+1);

for ix from 1 to n-j+1 do
  S[ix]:= DPhi(mult(R[1]^(ix-1),f,R),N,R);
 # print(ix,S[ix],ColumnDimension(S[ix]));
od:

for ix from n-j+2 to m+n-j-i+2 do
  #yea, confusing as mess index stuff I know.
    S[ix]:= DPhi(mult(R[1]^(ix-2+j-n),g,R),N,R);
 #  print(ix,S[ix],ColumnDimension(S[ix]));
od:

return S:

end proc;

#############################################################
#############################################################
#############################################################

#Proc:  RecoverCoFactors
#Input: vector over RR(t), diff algebra R
#Output: Ore Polynomials  over R[Dt,t] of the form [u,v]
#Purpose: Used to devectorize co-factors
RecoverCoFactors := proc(w,n,m,R)
local u,v;
u:=FromCoefficientVector(w[1..n+1],R[1]);
v:=FromCoefficientVector(w[n+2..m+n+2],R[1]);
return [u,v];
end proc;
#############################################################
#############################################################
#############################################################

#Proc:  ApproxCofactors
#Input: Ore polynomials f,g,h over diff algebra R where gcrd(f,g) ~ h w/ parameter epsilon
#			to search for noise
#Output: The right cofactors f*,g* such that f~f*h, g~g*h
#Purpose: Used to obtain the co-factors for f and g
#assumptions: We'll assume the input isn't too noisy. h is obtained via approx GCRD
ApproxCofactors:=proc(f,g,h,epsilon,R,threshold:=1e-15)
local m, #deg f
      n, #deg g
      d, #deg h
      i, #counter
      u, #bezout coef for f
      v, #bezout coef for g
      hbezout, #bezout gcrd. sort of like h.
      du, #n-d-1
      dv, #m-d-1
      lch, #leading coefficient of hbezout
      dsigma, #n-d
      dtau, #m-d
      Z, #used as a vector for solving linear systems
      W, #matrix used in conjunction with Z
      M, #M=M(adjoint(sigma),adjoint(tau))
      CofactorSystem, #
      lclmsys,#system used to solve for lclm coefs
      S,#diff sylvester matrix
      ff, #normalized f
      gg, #normalized g
      hh, #normalize h
      NSVector, #nullspace vector
      uvec, #uvector rep
      vvec; #vvector rep

  #set up degree bounds
  m:=degree(f,R[1]);
  n:=degree(g,R[1]);
  d:=degree(h,R[1]);
  
  ff:=f/OreNorm(f,R,2);
  gg:=g/OreNorm(g,R,2);
  hh:=h/OreNorm(h,R,2);
  
  #compute u and v by modified ExGCRD
  S:=DifferentialVandermonde(f,g,R);
  W:=DeleteColumn(S,1..d);
  Z:=UnitVector(1,ColumnDimension(W));
  NSVector:=LinearSolve(Transpose(W),Z,free='param');
  NSVector:=subs([seq(param[i]=0,i=1..m+n)],NSVector);
  #clean up higher-order term noise
  for i from n-d+1 to n do 
    NSVector[i]:=0; #clean up some noise
  od:
  for i from m-d+n+1 to m+n do 
    NSVector[i]:=0; #clean up the higher term noise
  od:
  for i from 1 to ArrayNumElems(NSVector) do
      NSVector:= simplify(NSVector*denom(NSVector[i])); #normalize the NS vector so it's in R[t]
  od;
  #remove numeric content via normalization
  NSVector:=NSVector/norm(Vector([seq(norm(expand(NSVector[i]),2),i=1..m+n)]),2);
  u:=FromCoefficientVector(NSVector[1..n],R[1]);
  v:=FromCoefficientVector(NSVector[n+1..m+n],R[1]);
  
  #get the leading coefficient of h and adjust u,v accordingly.
  #blow away the high order noise terms that appear here. 
  lch := CoefficientVector(simplify(mult(u ,ff,R)+mult(v,gg,R)),R[1]);
  for i from d+2 to ArrayNumElems(lch)  do 
    lch[i]:=0; 
  od:
  #these will need to be cleaned up w/ approximate GCD.
  uvec:=CoefficientVector(u,R[1])/lch[d+1];
  vvec:=CoefficientVector(v,R[1])/lch[d+1];
  
  for i from 1 to ArrayNumElems(uvec) do 
    uvec[i]:=NumericLowestTerms(uvec[i],R[2],threshold);
  od:
  for i from 1 to ArrayNumElems(vvec) do 
    vvec[i]:=NumericLowestTerms(vvec[i],R[2],threshold);
  od:
  u:=FromCoefficientVector(uvec,R[1]);
  v:=FromCoefficientVector(vvec,R[1]);
  
  #Next we compute sigma and tau...
  lclmsys:=GaussianElimination(Transpose(DifferentialSubResultant(ff,gg,R,d,d)));
  #manual correct. this matrix is rank defficient, and we are removing an
  #error term.
  #lclmsys[RowDimension(lclmsys)-1]:=Vector[row](ColumnDimension(lclmsys));
  
  return [u,v,lclmsys]
end proc;   
#############################################################
#############################################################
#############################################################

#Proc:  NumericRightDivision
#Input: Ore polynomials f,h over diff algebra R where h|f of the right
#Output: The quotient from right division, if it exists.
#Purpose: Used to obtain the co-factor for f
#assumptions: f and h have fractions cleared or this could get messy.
NumericRightDivision:=proc(f,h,R)
local m,
      d,
      fvec,
      fstar,
      Z,
      M,
      i;

#FIXME
#I know deleting columns after making the whole thing is bad, but this
#requires changing less code.

m:=degree(f,R[1]);
d:=degree(h,R[1]);
fvec:=DeleteRow(CoefficientVector(f,R[1]),1..d); 

#weird matrix...
M:=Matrix(m-d+1,m+1); 
for i from 1 to m-d+1 do 
  M[i]:=DPhi(mult(Dt^(i-1),h,R),m+1,R);
od:
M:=DeleteColumn(M,1..d);
fstar:=convert(LinearSolve(Transpose(M),fvec,free='param'),Vector);
#fstar;
#get rid of any parameters that come up
#fstar:=subs([param[1]=1,seq(param[i]=0,i=2..m+1)],fstar);
return FromCoefficientVector(fstar,R[1]);
end proc;


#############################################################
#############################################################
#############################################################
#Proc: NumericDivisionBasis
#Input: f,h over diff Algebra R
#Output: (m-d+1)x(m-d+1) reduced row basis matrix M(h)
#	 that corresponds to f~f^* h
#Purpose: Used to solve full rank system 
#Comments: Should be numerically robust b/c backwards substitution/Kramer solution.

NumericDivisionBasis:=proc(f,h,R)
local m,d,fvec,fstar,Z, M,i;


m:=degree(f,R[1]);
d:=degree(h,R[1]);
fvec:=convert(DeleteRow(CoefficientVector(f,R[1]),1..d),Vector); 
M:=Matrix(m-d+1,m+1);
for i from 1 to m-d+1 do 
  M[i]:=DPhi(mult(Dt^(i-1),h,R),m+1,R);
od:
M:=DeleteColumn(M,1..d);

return M;
end proc:

#############################################################
#############################################################
#############################################################

#Proc:  LCLMViaCoFactors
#Input: f,g over differential algebra R
#Output: LCLM of f and g according to uf+vg=0
#Purpose: Used to devectorize co-factors

LCLMviaCoFactors :=proc(f,g,epsilon,R)

local n,m,S,d,i,w,l,u,v,myNSV,lclmSystem,CofactorSystem;

n:=degree(g,R[1]);
m:=degree(f,R[1]);
S:=DifferentialVandermonde(f,g,R);
#m+n-deg(gcrd(f,g)) = deg(lclm(f,g))
d:=m+n- NumericRank(S,R[2],epsilon);
CofactorSystem:=DifferentialSubResultant(f,g,R,d,d);
#row-reduce the system
lclmSystem:=GaussianElimination(Transpose(CofactorSystem));
#works in exact case! fix for numerics later
#numeric hack!
lclmSystem[RowDimension(lclmSystem)] := Vector[row](ColumnDimension(lclmSystem)); #matrix is now rank deficient by 1
#really we want the nearest rank deficient matrix, but this is a little tricky...
myNSV:=op(NullSpace(lclmSystem));
w:=RecoverCoFactors(myNSV,n-d,m-d,R);
u:=w[1]; 
v:=w[2];

#vg is better numerically than uf for some reason...
return mult(v,g,R);
end proc;

#############################################################
#############################################################
#############################################################
#Proc: ExtendedApproxGCLD
#Input:
#Output:
#Purpose:
#General Comments: Computes the Approximate GCRD of f and g,
#		   along with bezout coefficients u and v
#Special Concerns:
#############################################################

#############################################################
#############################################################
#############################################################

#Proc: DifferentialCofactorSystem
#Input: Ore polynomials f,g of degree $m$ and $n$ respectively. 
#Output: (m+n-2d+2) X (m+n-2d) "extended" differential sylvester matrix
#Purpose: Used for setting up some systems of equations to compute co-factors

DifferentialCofactorSystem:=proc(f,g,R,m,n)
local S,ix,N,i;

N:=m+n;
#M:=(degree(f,R[1])+degree(g,R[1])
S:=Matrix(N,N);

for ix from 1 to m do
  S[ix]:= DPhi(mult(R[1]^(ix-1),f,R),N,R);
od:

for ix from m+1 to m+n do
  #yea, confusing as mess index stuff I know.
    S[ix]:= DPhi(mult(R[1]^(ix-1-m),g,R),N,R);
od:
return S:
end proc;
#############################################################
#############################################################
#############################################################
#Proc: ExtendedApproxGCRD
#Input: f,g over differnetial algebra R with search radius epsilon
#Output: [ApprodGCRD(f,g,epsilon,R),u,v] where uf+vg approx gcrd(f,g)
#Purpose: Compute Bezout coefficients (u,v) for (f,g). 
#General Comments: Computes the Approximate GCRD of f and g,
#		   along with bezout coefficients u and v
#Special Concerns:O
ExtendedApproxGCRD:=proc(f,g,epsilon,R)
  local S,m,n,d,i,W,Z,NSVector,VectorizedGCRD,param,u,v;
  
  #normalize f and g first
  S:=DifferentialVandermonde(f/OreNorm(f,R,2),g/OreNorm(g,R,2),R);
  #compute the numeric Rank
  #verify this step
  m:=degree(f,R[1]);
  n:=degree(g,R[1]);
  d:=m+n-NumericRank(S,R[2],epsilon);
  print("is the degree of gcrd",d);
  #delete d columns. This gives us a system we can solve!
  #if d=0 then nothing happens
  W:=DeleteColumn(S,1..d);
  
  #prevents total annihilation
  Z:=Vector(ColumnDimension(W));
  Z[1]:=1; 
 
  #I am assuming this gives us the "low degree" canonical
  #solution... It *should*, but I haven't proved this yet.
  NSVector:=LinearSolve(Transpose(W),Z,free='param');
  
   print("NSV w/ params:",NSVector);
  #set the first parameter to be non-zero.
  NSVector:=subs([seq(param[i]=0,i=1..m+n)],NSVector);
 # param[1]:=1;
  for i from n-d+1 to n do 
    NSVector[i]:=0; #clean up some noise
  od:
  for i from m-d+n+1 to m+n do 
    NSVector[i]:=0; #clean up the higher term noise
  od:
  
  for i from 1 to ArrayNumElems(NSVector) do
      NSVector:= normal(NSVector*denom(NSVector[i])); #normalize the NS vector so it's in R[t]
  od;
  #some normalization
   print("NSV after cleanup:",NSVector);
  NSVector:=NSVector/norm(Vector([seq(norm(expand(NSVector[i]),2),i=1..m+n)]),2);
  #now we basically have one vector in the null space without any shennanigans. 
  #print('NSV',NSVector);
  
  VectorizedGCRD:=Transpose(S).NSVector;
  #print(VectorizedGCRD);
  #make leading coefficient polynomial
  #VectorizedGCRD:=VectorizedGCRD*denom(VectorizedGCRD[d+1]);
  
  #get rid of leftover noise
  for i from d+2 to ArrayNumElems(VectorizedGCRD) do
    VectorizedGCRD[i]:=0; 
  od:
  #print("GCRD",VectorizedGCRD);
  #deep copies
  u:=collect(simplify(FromCoefficientVector(Vector(NSVector[1..n]),R[1]),R[2]),R[1]);
  v:=collect(simplify(FromCoefficientVector(Vector(NSVector[n+1..m+n]),R[1]),R[2]),R[1]);
  
  #print('NSV',NSVector);
 # print('v',v);
#  print('u',u);
  return [FromCoefficientVector(expand(simplify(VectorizedGCRD),R[2]),R[1]),
	 u,v];

  
	 
  #case S has full rank
  #use kramer or inverse here.
  
  
  #case S is rank deficient
  #solve under constrained linear syste
end proc;

#############################################################
#############################################################
#############################################################


#############################################################
#############################################################
#############################################################
#Proc: CofactorApproxGCRD
#Input: 
#Output:
#Purpose:
#General Comments:
#Special Concerns

#############################################################
#############################################################
#############################################################
#############################################################
#############################################################
#############################################################
#Proc: NumericContentNormalization
#Input: Orepolynomial f over differential algebra R
#Output: f/numeric_content(f)
#Purpose: estimates size of the numeric content of f
#General Comments: not rigorous, just sort of a guess.
#Special Concerns: 0 polynomial is weird.
NumericContentNormalization:=proc(f,R)

local v,
      mymaxnum,
      mymaxdenom,
      i;

v:=CoefficientVector(collect(f,R[1]),R[1]);
mymaxnum:=1;
mymaxdenom:=1;
for i from 1 to ArrayNumElems(v) do
    mymaxnum:=max(norm(expand(numer(normal(v[i]))), 2),mymaxnum);
    mymaxdenom:=max(norm(expand(denom(normal(v[i]))), 2),mymaxdenom);
od:
expand(mymaxnum/mymaxdenom)
end proc;


#############################################################
#############################################################
#############################################################


#Proc ExpandRowVector
#Input: vector in R^(1 x n)[x] whose degree is bounded above by mu
#Output: Vector in R^(1 x mu*n) whose entries are expanded "blocks" of the input. 
#Expands a row vector of dimension 1xn to 1 x mu*n, padding with zeros where appropriate. 

#Notes on Usage: I designed the function to take mu in as an argument, opposed to deriving it
#		because this gives us more direct control on the sizes of the resulting vector,
#		should we decide we screwed up some where
ExpandRowVector := proc(w,x, mu)

local i,j,n,wi,what; #"double-u hat"

n :=ArrayNumElems(w); #number of things in the vector w. 
what:= Vector[row](n*(mu));
for i from 1 to n do #iterate through each polynomial in the row vector w, of which there be n
  wi:= CoefficientVector(w(i),x); #extract coefficients, make sure they are in the right order. They should be w0,w1,w2,... wn-1
  for j from 1 to ArrayNumElems(wi) do 
    what((i-1)*(mu) + j) := wi(j); 
  od;
od;
what;
end proc;

#############################################################
#############################################################
#############################################################

#Proc: ApproxGCRDDegree(f,g,epsilon,R,d)
#Input: f,g in R[Dx,x,;] with known degree d of the GCRD.
#Output: Approximate GCRD of f and g with tolerance epsilon
#This function just mashes a bunch of other functions together
#to get the GCRD;

ApproxGCRDDegree := proc(f,g,epsilon,R,d)
local V,Gd,VectorizedGCRD,i,myGCRD,Vrank;

V:=DifferentialVandermonde(f,g,R);#compute the Differential 
				 #Sylvester-like/Vandermonde Matrix for our computations
#Vrank:=NumericRank(V,R[2],epsilon); #store the rank here.
#if(Vrank = fail) then 
#    return fail;
#else
    Gd := d; #the degree of the GCRD by SVD
    myGCRD:=0; #for reconstruction purposes we set it to 0 and manually handle the special cases

    if (f=0 and g=0) then
	return 0; #trivial GCRD
    elif (f=0 and g<>0 ) then
	return evalf(g); #trivial GCRD
    elif (f<>0 and g =0)
	then return evalf(f); #trivial GCRD
    #GCRD is unit therefore we are done.
    elif (Gd = 0) then  #this means that our original matrix has rank 1
	return 1 ; #trivial GCRD; 

    else
	VectorizedGCRD:=ComputeGCRDVector(V,Gd,R,f,g);
	
	#re-construct the polynomial from the vector carefully. 
	#I think I need to use mult because of the non-commutative structure.
	for i from 1 to Gd+1 do #changed it so it ignores values that should be 0
	    #up to Gd because everything else should be 0
	    
	    #multiply so the coefs are on the left
	    #I don't trust CoefficientVector()...
	    myGCRD:=myGCRD+mult(VectorizedGCRD(i),R[1]^(i-1),R);
	od;
    fi;
# fi;
  if(myGCRD=0) then return FAIL fi;
  return myGCRD;
end proc;


#############################################################
#############################################################
#############################################################

#Proc: UnstructuredApproxGCRD(f,g,epsilon,R)
#Input: f,g in R[Dx,x,;] with approximation epsilon
#Output: Approximate GCRD of f and g with tolerance epsilon
#Unstructured approach to to perturbing f and g to get a GCRD.

UnstructuredApproxGCRD := proc(f,g,epsilon,R)

local i,BlockSize,myRank, V,n,d,mu,Big,Ul,Sl,Vtl,SS,Bhat,fvec,gvec,fnew,gnew,gnewAvg,fnewAvg,myBhat; #a lot of vars...

#this is because when I wrote the differential vandermonde code I was lazy and put the lower degree term first instead of handling two cases
# when I fixed a bug in the code. we need this for reconstruction purposes anyways.

if(degree(f,R[1])> degree(g,R[1])) then return UnstructuredApproxGCRD(g,f,epsilon,R) fi;

V:=DifferentialVandermonde(f,g,R);
n:=RowDimension(V);
d:=MatrixDegree(V,R[2]);
mu:=2*n*d;

Big:=ExpandDifferentialVandermondeMatrix(V,R[2],mu,d,n);

Ul,Sl,Vtl:=SingularValues(Big, output = ['U', 'S', 'Vt']); #get singular values and matrices for stuff and reconstruction.
###WARNING###:  Ul.S.Vtl ~=~ Big, so this will cause some problems with the GCRD. it will be perturbed "a bit".

BlockSize:=ColumnDimension(Big)/ColumnDimension(V); #Singular values are distributed in column blocks

myRank:=ScaledCTGWRank(Sl,epsilon,RowDimension(Big),ColumnDimension(Big),mu,n,d); #use the ScaledCGTWRank based on the column block size

#do some tweaking of the indices to account for the ColumnDimension(Big)-RowDimension(Big) singular values that are zero and omitted.
for i from RowDimension(Sl) -(RowDimension(V) - myRank)*BlockSize+ColumnDimension(Big)-RowDimension(Big)+1 to RowDimension(Sl) do
      Sl[i]:=0;
od;

SS := DiagonalMatrix(Sl,RowDimension(Big),ColumnDimension(Big));
Bhat:= Ul.SS.Vtl; #compute the new Blown Up "perturbed" Vandermonde matrix.

#compute the degree vectors for f and g in the reconstruction, based on the input parameters
fvec:=DegreeVector(f,R);
gvec:=DegreeVector(g,R);


#weighted recovery of f and g from the perturbed matrix
fnewAvg:=0;
gnewAvg:=0;
myBhat:=Bhat;
for i from 1 to mu+1 do
 fnewAvg:=fnewAvg + RecoverFromRowVector(Row(myBhat,i),R,fvec,degree(f,R[1]),ColumnDimension(Big)/RowDimension(V),i-1);
 gnewAvg:=gnewAvg + RecoverFromRowVector(Row(myBhat,degree(g,R[1])* (mu+1)+i),R,gvec,degree(g,R[1]),ColumnDimension(Big)/RowDimension(V),i-1);
od;

fnewAvg:=collect(evalf(fnewAvg/(mu+1)),R[1]);
gnewAvg:=collect(evalf(gnewAvg/(mu+1)),R[1]);

#recover f and g from the perturbed matrix

fnew:=RecoverFromRowVector(Row(Bhat,1),R,fvec,degree(f,R[1]),ColumnDimension(Big)/RowDimension(V),0);
gnew:=RecoverFromRowVector(Row(Bhat,degree(g,R[1])* (mu+1)+1),R,gvec,degree(g,R[1]),ColumnDimension(Big)/RowDimension(V),0); #we need to compute an offset +1 for g

#solve some equations to get an approximate GCRD.
return [ApproxGCRDDegree(fnew,gnew,epsilon,R,RowDimension(V) - myRank),fnew,gnew,fnewAvg,gnewAvg,Bhat] ; #return a list of the quantities we computed

end proc;

#############################################################
#############################################################
#############################################################

#Proc: ApproxGCRD(f,g,epsilon,R)
#Input: f,g in R[Dx,x,;]
#Output: Approximate GCRD of f and g with tolerance epsilon
#This function just mashes a bunch of other functions together
#to get the GCRD;



ApproxGCRD := proc(f,g,epsilon,R)
local V,Gd,VectorizedGCRD,i,myGCRD,Vrank;

V:=DifferentialVandermonde(f,g,R);#compute the Differential 
				 #Sylvester-like/Vandermonde Matrix for our computations
Vrank:=NumericRank(V,R[2],epsilon); #store the rank here.
print(Vrank);
if(Vrank = fail) then 
    return fail;
else
    Gd := RowDimension(V) - Vrank; #the degree of the GCRD by SVD
    myGCRD:=0; #for reconstruction purposes we set it to 0 and manually handle the special cases

    if (f=0 and g=0) then
	return 0; #trivial GCRD
    elif (f=0 and g<>0 ) then
	return evalf(g); #trivial GCRD
    elif (f<>0 and g =0)
	then return evalf(f); #trivial GCRD
    #GCRD is unit therefore we are done.
    elif (Gd = 0) then  #this means that our original matrix has rank 1
	return 1 ; #trivial GCRD; 

    else
	VectorizedGCRD:=ComputeGCRDVector(V,Gd,R,f,g);
	#re-construct the polynomial from the vector carefully. 
	#I think I need to use mult because of the non-commutative structure.
	for i from 1 to Gd+1 do #changed it so it ignores values that should be 0
	    #up to Gd because everything else should be 0
	    
	    #multiply so the coefs are on the left
	    #I don't trust CoefficientVector()...
	    myGCRD:=myGCRD+mult(VectorizedGCRD(i),R[1]^(i-1),R);
	od;
    fi;
  fi;
  
  if(myGCRD = 0) then return FAIL fi;
  return myGCRD;
end proc;



#############################################################
#############################################################
#############################################################

#Proc: RemoveContent
#Input: Vector C over R(x)
#Output: Vector over R[x] with polynomial content removed, if possible. 
#Used in the ComputeGCRDVector to remove the content and get the primitive part of the GCRD

RemoveContent:=proc(C,x,d)
  if ContentRemoval = true then 
      return ApproxDivide(C,d,x);
  fi;
  return C;
end proc;
#############################################################
#############################################################
#############################################################

#Proc: ComputeGCRDVector
#Input: Differential Vandermonde matrix V, degree of GCRD d, differential algebra R
#	and Diff polys f and g.
#Output: Vectorized GCRD of f and g if it exists
#Used in the ComputeGCRDVector to remove the content and get the primitive part of the GCRD

ComputeGCRDVector := proc(V,d,R,f,g)
local W,NSVector, VectorizedGCRD,Z,s, temp,x,i;
x:=R[2]; #assign the variable from the differential algebra before I forget.

W:=ReduceColumnDifferentialVandermondeMatrix(V,d+1); #d+1 because we need d+1 spots for G=GCRD(f,g);
Z:= Vector(ColumnDimension(W));
NSVector := LinearSolve(Transpose(W),Z,free='s'); #grab the first NullSpace vector and call it good.
						#Really I think there should only be 1, and it doesn't matter because
						#we are just "blowing away" the high order terms.
for i from 1 to RowDimension(NSVector)-1 do 
s[i]:=1; #blow up stuff in the null space we don't care about. Just need one vector.
	#perhaps an inteligent choice of solving the system  or parameter selection would limit the 
	#overlying field GCD computations. 
	#I am guessing a poorly conditioned matrix might cause some problems here.
	#perhaps a better way to select which parameters I don't destroy would be
	#the one that is furtherest/closest to 0 with some norm.
od;
#set the last param to be 1

s[RowDimension(NSVector)]:=1;
for i from 1 to ArrayNumElems(NSVector) do
    NSVector:= NSVector*denom(NSVector(i)); #normalize the NS vector so it's in R[t]
od;
#now we basically have one vector in the null space without any shennanigans. 

VectorizedGCRD:= Transpose(V).NSVector;
			     #in content removal and normalization.
			     #this will give us a vectorized form of the 
			     #GCRD. We need to normalize and remove content
			     #since we have a multiple of the GCRD over R(t).
			     #This will involve commutative GCD computations over
			     #R[t] which may be slow...

			     
#slow implementation of computing the content of the coefficients. 
#faster to use a logarthmic order D&C algorithm, but this is a prototype alg... 
#Let's just hope it's not too slow

#this is just checking some simple cases
#we aren't using maples gcd() command for numeric stuff because it's bad.
#try
    if(degree(lcoeff(f,R[1]),R[2]) = 0 or degree(lcoeff(g,R[1]),R[2]) = 0) then
	#WARNING("Tryed to removed content W/ fft");
	#return(ApproxDivide(VectorizedGCRD,d,R[2]));

    fi;
#catch: WARNING("Unable to remove content");
    
#end try;
					#get rid of the content, we just want the primitive part
					#This might be faster if we converted it to a polynomial and used 
					#Content() or somethng #I am taking out content removal for now.
return (VectorizedGCRD);
end proc;

#############################################################
#############################################################
#############################################################


#Proc: ReduceColumnDifferentialVandermondeMatrix - better name?
#Input: V of dimension n x n over R(t) and Rem, an integer representing the number of columns to remove
#       The other choice is to remove Rem+1 columns, but this may be confusing so it is designed in this way to remove "0" columns
#       and effectively perform a deep copy.
#Output: New matrix V~ that has the first Rem columns removed
#Used for solving the system for the n-d-1 zeros in  (u,v)V(f,g) = (G,0)^T where G = GCRD(f,g), deg(G) = d.

ReduceColumnDifferentialVandermondeMatrix := proc(V,Rem)
#I should use BlockCopy() but this is an experimental version and it's slightly confusing
local i,j,W,n,m;

n:=RowDimension(V);#compute Row Dimension; will use alot.
m:=ColumnDimension(V)-Rem; #compute Col Dimension; will use alot.

W:=Matrix(n,m); #n x n-Rem matrix;

for i from 1 to n do
  for j from 1 to m do 
  W(i,j) := V(i,j+Rem); 
  od;
od;

end proc;


#############################################################
#############################################################
#############################################################
#Proc RecoverFromRowVector
#Input: A row vector of Vbig, differential Algebra R, degree of the function to recover
#       in degreeVector(R[1]) and R[2], and the block size of the columns, with column offset.
#Output: A differential polynomial recovered from the vector.

RecoverFromRowVector := proc(MyRow,R,dVector,D,BlockSize,offset)
local functionVector,i,j,temp,currentBlock;
functionVector:=Vector(D+1); #create a vector to recover our polys
currentBlock:=1;
for i from 1 to D+1 do 
    if dVector[i] >=0 then  #-infty case, skip it.
	temp:=Vector(dVector[i]+1);
	for j from currentBlock to currentBlock+dVector(i) do#The +1 is because there are d+1 spots
	    temp[1+j-currentBlock]:= MyRow[j+offset];
	od;
	functionVector[i]:=FromCoefficientVector(temp,R[2]); #vector in t
    fi;
	currentBlock:=currentBlock+BlockSize;
    
od;
return collect(FromCoefficientVector(functionVector,R[1]),R[1]);
end proc;

#############################################################
#############################################################
#############################################################

#Proc ExpandDifferentialVandermondeMatrix
#Input: V in R^(n x n)[x] whose entries are bounded above by d and a parameter mu bounding the degree of null space of V
#Output: Block matrix (See Cauchy matrix) representation of V in R^(n*(mu+1) x n*(mu + d+ 1) 
#Used for converting the differential Vandermonde matrix into a block-scalar form so we can do SVD

ExpandDifferentialVandermondeMatrix :=proc (V,x,mu,d,n)

local Vhat,i,j,k, VhatRow;

Vhat := Matrix(n*(mu+1),n*(mu+d+1));

for i from 1 to n*(mu+1) do 
  
 VhatRow := ExpandRowVector(x^( modp(i-1,mu+1))*Row(V,ceil(i/(mu+1))), x, (d+mu+1)); 
 
 #this seems complicated so it merits an explanation
 #we are iterating through the n*(mu+1) rows of the matrix.
 #on row ceil(i/(mu+1))  of V we are creating row i of Vhat.
 #Row i consists of row ceil(i/(mu+1)) shifted  i mod (mu+1) positions - 0,1,2,...,mu
 
 
 #copy some stuff over. probably faster to use the built in maple routines, screw it.
  for j from 1 to n*(d+mu+1) do 
    Vhat(i,j) := VhatRow(j);
  od;
od;
Vhat;
end proc;
#############################################################
#############################################################
#############################################################
#Proc: NumericRank
#Input: n x n matrix over R[x] named V, indeterminant x and error tolerance eps (epsilon)
#Output: Numeric rank of M, as determined by the comSVD.
#Used to compute the numeric rank of M, which is the scaled rank
#of the Cauchy matrix representation of the DifferentialVandermonde matrix, V. 


NumericRank := proc (V,x,epsilon)
local Vbig,mu,n,d, Sigma,i,k1,k2,k3,k,m,Delta;

d:=MatrixDegree(V,x);
n:=RowDimension(V);
mu:=2*n*d;
m:=ColumnDimension(V);
Delta := evalf(sqrt(m+n));

Vbig := ExpandDifferentialVandermondeMatrix(V,x,mu,d,n); #form the Cauchy matrix for V
Sigma := SingularValues(evalf(Vbig));	#compute singular values
					#evalf() first speeds up the computation - that is compute SVs numerically instead of algebreically
					#If done algebreically, complex roots may be introduced and messed up my computations pretty badly.

k1:=ScaledCTGWRank(Sigma,epsilon,RowDimension(Vbig),ColumnDimension(Vbig),mu,n,d);
#k2:=EpsilonRank(Sigma,epsilon,RowDimension(Vbig),n);
#k3:=RatioRank(Sigma,epsilon,RowDimension(Vbig),n);

return k1;
end proc;

#############################################################
#############################################################
#############################################################

#MISC HELPER FUNCTIONS
#some functions just to do stuff that needs to be done
#Norms, transforms and general utility functions

#############################################################
#############################################################
#############################################################


#Proc PrettyPrint
#Input: Diff poly f over R[Dt,t]
#Output: sort collected components
#Used to compute "d" in our matrix computations
PrettyPrint :=proc(f,R)
  local fvec,
	i,
	my_ans:=0 #maples sort routine breaks our answer, so we need
	          #extract the polynomial ourselves. 
	;
  fvec:=CoefficientVector(f,R[1]); 
  for i from 1 to ArrayNumElems(fvec) do
    fvec[i]:=sort(fvec[i],R[2]); #sort components by `t`
  od:
  #implicitly sorted	
  for i from  ArrayNumElems(fvec) to 1 by -1 do 
    my_ans:= my_ans+ fvec[i]*R[1]^(i-1);
  od:
  return (my_ans);
end proc;


#############################################################
#############################################################
#############################################################
#Proc MatrixDegree
#Input: Matrix M over R^(m x n)[x] 
#Output: Degree of the matrix
#Used to compute "d" in our matrix computations
MatrixDegree :=proc(M,x)
local i,deg := -1; 
for i from 1 to ArrayNumElems(M) do 
  deg := max(deg,degree(M(i)));
od;

end proc;
#############################################################
#############################################################
#############################################################

#Proc: OreNorm
#Input: Ore Polynomial in Differential Algebra R = R[Dx,x] and parameter p
#Output: "Super p norm" of the polynomial. The p-norm of the p-norm of each polynonmial orver R[x] coefficients.
#Used for measuring the "size" of a polynomial
OreNorm := proc(f,R,p)
local myNorm, coefficients,i;

coefficients := CoefficientVector(f,R[1]); #get coefficients in Dx

myNorm := Vector(ArrayNumElems(coefficients));

for i from 1 to ArrayNumElems(coefficients) do
    myNorm(i) := norm(CoefficientVector(coefficients(i),R[2]),p); #ith norm	
od;
return(norm(myNorm,p));
end proc;
#############################################################
#############################################################
#############################################################
#Proc ApproxDivideQuo
#Input: diff poly whose content is the leading term
#Output:   content used by dropping remainders
#This is used to remove extra content, should floating point problems prevent a division.

ApproxDivideQuo :=proc(G,R)

local Intermediate,i,j,n,omega,Solution,a,b,A,B,Threshold,Gvec,d,C,k;
Threshold:=1e-8; #small enough for our examples

Gvec:=CoefficientVector(convert(G,rational),R[1]);
Solution:=Vector(ArrayNumElems(Gvec));
d:=degree(G,R[1]); #Dt degree

for i from 1 to ArrayNumElems(Gvec)-1 do
  Solution[i]:=quo(Gvec[i],Gvec[d+1],R[2]); #drop remainders
od;

Solution(d+1):=1;#this is 1 no matter what

return evalf(collect(FromCoefficientVector(Solution,R[1]),R[1]));
end proc;

#############################################################
#############################################################
#############################################################
#Proc: Round
#Input: Flop to round down and the precision
#Output: Rounded flop.
Round := proc(x,n::integer:=1)
   parse~(sprintf~(cat("%.",n,"f"),x)); 
end proc:

#############################################################
#############################################################
#############################################################
#Proc ApproxDivideFFT
#Input: Vector consisting of polynomials and index d where G[d+1] approximately
#       divides every element.
#Output:  degree(f)-degree(g)+1 polynomial approximation of f(t)/V[d+1]
#This is used to remove extra content, should floating point problems prevent a division.

ApproxDivideFFT :=proc(G,R,threshold:=1e-9)

local Intermediate,i,j,n,omega,Solution,a,b,A,B,Gvec,d,C,k,temp;
#Threshold:=1e-14; #small enough for our examples

#get rid of some noise
#normalize first
Gvec:=CoefficientVector(G/OreNorm(G,R,2),R[1]);



Solution:=Vector(ArrayNumElems(Gvec));
d:=degree(G,R[1]); #Dt degree

for i from 1 to ArrayNumElems(Gvec)-1 do
      omega:=degree(Gvec[i],R[2]); #compute omega-1-PRU
      a:=CoefficientVector(Gvec[i],R[2]);
      b:=CoefficientVector(Gvec[d+1],R[2]);
      b:=Vector(b,omega+1);
      
      A:=FourierTransform(a,normalization=none); #important not to normalize
      B:=FourierTransform(b,normalization=none); #important not to normalize
      C:=Vector(ArrayNumElems(A),j->A[j]/B[j]); #This is the un-normalized FFT representation of f and g
      Intermediate:=  Re( InverseFourierTransform(C,normalization=none))/(omega+1);  #normalize here
      for k from 1 to ArrayNumElems(Intermediate) do
	  if (abs(Intermediate[k]) < threshold) then Intermediate[k] :=0; fi;
	  if(k>  (degree(cleanup(Gvec[i],R[2],threshold),R[2]) - degree(cleanup(Gvec[d+1],R[2],threshold),R[2]) + 1)) then 
	      Intermediate[k]:=0 #blow away high order terms that should be zero.
	  fi;
      od;
      Solution[i]:=collect(FromCoefficientVector(Intermediate,R[2]),R[2]);
      
od;

Solution(d+1):=1;#this is 1 no matter what
return collect(FromCoefficientVector(Solution,R[1]),R[1]);
end proc;
#############################################################
#############################################################
#############################################################
#Proc: NumericQuo
#Input: a,b over R[t]. Implicit thresholding is to 15 decimal places and
#	p-norm with p=2 is used as default.
#Output:   the quotient of a/b
#Purpose: used to get rid of numeric content
#Comments: this won't work if b does not approximately divide a
#	 
NumericQuo:=proc(a,b,t,threshold:=1e-15,p:=2)
local avec,
      bvec,
      i,
      aa,
      bb,
      omega, #Omega PRU
      aavec,
      bbvec,
      A, #FFT representation of a
      B, # FFT represtation of b
      C, #FFT of a/b
      Intermediate; #intermediate result from FFt
      
avec:=evalf(CoefficientVector(a/norm(expand(a,t),p),t));
bvec:=evalf(CoefficientVector(b/norm(expand(b,t),p),t));

#clean up the noise from a
for i from 1 to ArrayNumElems(avec) do 
  if abs(avec[i]) < threshold then
      avec[i]:=0;
  fi;
od:
#clean up the noise from b
for i from 1 to ArrayNumElems(bvec) do 
  if abs(bvec[i]) < threshold then
      bvec[i]:=0;
  fi;
od:

#cleaned up a and b
aa:=FromCoefficientVector(avec,t);
bb:=FromCoefficientVector(bvec,t);
#omega PRU
omega:=degree(aa,t);
#we may have blown away the high order degree, so we need to re-consider our data to
#prevent miscalculation of degrees.
aavec:=CoefficientVector(aa,t);
bbvec:=Vector(CoefficientVector(bb,t),omega+1);


A:=FourierTransform(aavec,normalization=none); #important not to normalize
B:=FourierTransform(bbvec,normalization=none); #important not to normalize
#C:=Vector(ArrayNumElems(A),j->A[j]/B[j]); #This is the un-normalized FFT representation of a and b
C:=Vector(ArrayNumElems(A));
for i from 1 to ArrayNumElems(A) do 
  if(B[i] <> 0) then 
    C[i]:=A[i]/B[i];
  fi;
od:
Intermediate:=  Re( InverseFourierTransform(C,normalization=none))/(omega+1);  #normalize here

#clean up the recovered polynomial
for i from 1 to ArrayNumElems(Intermediate) do 
  if(abs(Intermediate[i])<threshold) then
    Intermediate[i]:=0;
  fi:
  if(i>degree(aa,t)-degree(bb,t)+1) then 
    Intermediate[i]:=0 # blow away high order terms
  fi:
od:

return FromCoefficientVector(Intermediate/norm(Intermediate,p),t);
end proc;
#############################################################
#############################################################
#############################################################
#Proc: cleanup
#Input: rational function a in indeterminant t w/ threshold
#Output: rational function w/less noise
#Purpose: fix function so we can use approx gcd

cleanup:=proc(a,t,threshold:=1e-15,p:=2)
local mynum,mydenom,i;

#catch this right away
if(a=0) then return 0 fi;
mynum:=CoefficientVector(expand(numer(a)),t);
#mynum:=mynum/norm(mynum,p);
mydenom:=CoefficientVector(expand(denom(a)),t);
#mydenom:=mydenom/norm(mydenom,p);

for i from 1 to ArrayNumElems(mynum) do 
  if(abs(mynum[i])<threshold) then mynum[i]:=0 fi;
od:
for i from 1 to ArrayNumElems(mydenom) do 
  if(abs(mydenom[i])<threshold) then mydenom[i]:=0 fi;
od:

if(norm(mynum,p) = 0) then return 0 fi;
if(norm(mydenom,p)=0) then return FAIL fi;

return FromCoefficientVector(mynum,t)/FromCoefficientVector(mydenom,t);
end proc;

#############################################################
#############################################################
#############################################################
#Proc: NumericDifferentialLowestTerms
#Input: Diff poly over R[Dt,t] with rat function coefficients
#Output: Diff poly over R with rat function coefficients, that
#	 are reduced by taking approximate GCD.
#Purpose: Used to clean up the messes we get from doing divisions
#	 and approximate GCRD.
NumericDifferentialLowestTerms :=proc(f,R,threshold:=1e-15,p:=2,exponent:=3)

  local fvec, #vectorized version of f for performing arithmetic
	i;

  fvec:=CoefficientVector(expand(f,R[2]),R[1]);

  for i from 1 to ArrayNumElems(fvec) do 
      fvec[i]:=NumericLowestTerms(fvec[i],R[2],threshold,p,exponent);
  od:
return FromCoefficientVector(fvec,R[1]);
end proc:



#############################################################
#############################################################
#############################################################
#Proc: NumericLowestTerms
#Input: rational function a, in indeterminant t with tolerance
#Output: rational function approximately a in least terms
#Purpose: Used to clean up messy rational functions from linear algebra

NumericLowestTerms:=proc(a,t,threshold:=1e-15,p:=2,exponent:=3)
 local g,mydenom,mynum,aa;
  #use approximate GCD
  #QRGCD seems to perform well in practice
  #Kaltofen et als gcd algorithm works better for high degree input.
  aa:=expand(numer(a))/expand(denom(a));
  aa:=cleanup(aa,t,threshold);
  if(denom(aa) = 1) then return aa fi;
  #return 0 polynomial
  if(aa = 0) then return 0  fi;
  
  g:=[SNAP:-QRGCD(numer(aa),denom(aa),t,10^(-1*exponent))][3];
 # g:=ugcd(numer(aa),denom(aa),t,exponent);
  
  #divide out the approximate content
  mynum:=NumericQuo(numer(aa),g,t,threshold);
  mydenom:=NumericQuo(denom(aa),g,t,threshold);

  #check for zero denom
  #we check because thresholding could blow away a small term.
  if(mydenom = 0) then 
    WARNING("Zero Denominator in reducing order");
    return 0;
  fi; 

  #re-scale the proportions of a.  This prevents associates of a
  #from appearing and breaking computations.
  return evalf( norm(expand(numer(aa)),p)*mynum )/evalf(norm(expand(denom(aa)),p) * mydenom);
end proc;


#############################################################
#############################################################
#############################################################
###DEPRECATED DO NOT USE###
#Proc ApproxDivide
#Input: Vector consisting of polynomials and index d where W[d+1] approximately
#       divides every element.
#Output:  degree(f)-degree(g)+1 polynomial approximation of f(x)/V[d+1]
#This is used to remove extra content, should floating point problems prevent a division.

#Proc ApproxDivide
#Input: Vector consisting of polynomials and index d where W[d+1] approximately
#       divides every element.
#Output:  degree(f)-degree(g)+1 polynomial approximation of f(x)/V[d+1]
#This is used to remove extra content, should floating point problems prevent a division.

ApproxDivide :=proc(W,d,x)

local V,i,j,n,Solution,Intermediate,Threshold;
Threshold:=1e-10; #small enough for our examples
V:=collect(W,x);#this stops maple from breaking some computations but it is expensive
Solution:=Vector(ArrayNumElems(V));
for i from 1 to d do 
    n:=degree(collect(V(i),x),x)+1-degree(collect(V(d+1),x),x);
    
    if n <= 0 then 
     #print(V(i),V(d+1));
     WARNING("Unable to remove content or there is no content");
     return(W);
    fi;
    #Very slow and possibly unstable fourier transform, using the same default
    #normalization factor as maple
    #The normal thing is incase we have a zero appearing or something...
    #Intermediate:=1/sqrt(n) * Vector(n,j->subs(x=exp((2*Pi*I*(j-1))/n),V(i)/V(d+1)));
        #invert the fourier transform and store the approximation to the rational function
    #Solution(i):=FromCoefficientVector(Re(InverseFourierTransform(Intermediate)),x);
    #
    #
    #let's try using Quo...
    if norm(rem(W[i],W[d+1],x),2) > Threshold then return(W) fi;
    
    Solution(i):= quo(W[i],W[d+1],x);
    
od;
Solution(d+1):=1;#this is 1 no matter what
return Solution;
end proc;

#############################################################
#############################################################
#############################################################
#Proc: Noise
#Input: noise factor delta and polynomial f over differential Algebra R
#Output: suitable Delta(f) term for experiments
#This is used for creating a Delta(f). The goal is to add a little noise to a bit of everything
#and hope it won't mess things up too bad.

Noise := proc(f,delta,R)
local D,d,noise;

D:=degree(f,R[1]); #Dt degree
d:=degree(f,R[2]); #t degree

noise:= RandDPoly(d,D,R);
noise := noise/evalf(OreNorm(noise,R,2)); #scale the noise down!
noise:=noise*delta; #make the noise have L2 norm <delta

end proc;

#############################################################
#############################################################
#############################################################
#Proc: RandDPolyUnormalized
#input: degree d for coefficients and degree D for the differential operator over the differential algebra R.
#Output: Differential polynomial f over R with coefficient degrees bounded by d and Differential deree D with coefficients without normalization


RandDPolyUnormalized := proc(d,D,R)
local V,i;
V:=Vector(D+1); #create poly w/ D elements

for i from 1 to D do
    V[i] := randpoly(R[2],degree = d, sparse); #make it sparse in order to save computation time. 
od;
V[D+1] :=1 ; #make it monic for now. We like monic stuff.

return FromCoefficientVector(V,R[1]); #this should give us what we want.

end proc;


#############################################################
#############################################################
#############################################################
#Proc: RandDPoly
#input: degree d for coefficients and degree D for the differential operator over the differential algebra R.
#Output: Differential polynomial f over R with coefficient degrees bounded by d and Differential deree D with coefficients


RandDPoly := proc(d,D,R)
local V,i;
V:=Vector(D+1); #create poly w/ D elements

for i from 1 to D do
    V[i] := randpoly(R[2],degree = d, sparse); #make it sparse in order to save computation time. 
    V[i] := V[i]/evalf(norm(V[i],2)); #normalize them to stop the OreNorm from getting stupid
od;
V[D+1] :=1 ; #make it monic for now. We like monic stuff.

return FromCoefficientVector(V,R[1]); #this should give us what we want.

end proc;

#############################################################
#############################################################
#############################################################
#Proc: DifferentialConvolutionMatrix
#Input: Ore Polynomial f and degree d of the convolution with differential algebra R
#Output: Matrix of the form (f, \Df, \D^2 f, ..., \D^d f)
#Purpose: Used to set up a linear system to compute co-factors
DifferentialConvolutionMatrix:=proc(f,d,R)

local df,M,i;
df:=degree(f,R[1]);
M:=Matrix(d+1,df+d+1);
df:=degree(f,R[1]);
for i from 1 to d+1 do
  M[i]:=DPhi(mult(R[1]^(i-1),f,R),df+d+1,R);
od:
M;
end proc:

#############################################################
#############################################################
#############################################################
#Proc: DifferentialBezoutCoefficients
#Input: Ore Polynomials f,g,h over differential algebra R where h=gcrd(f,g)
#Output:[u,v] the Bezout coefficients to uf+vg=h
#Purpose: Used to compute the bezout coefficients of minimal degree
#	 for approximate LCLM


#############################################################
#############################################################
#############################################################
#############################################################
##RANK FUNCTIONS###
#This section contains some functions used for determing the numeric rank of a matrix using different algorithms.
#they all take an n x 1 matrix consisting of sinuglar values and try and guess the rank based on them. 

#Proc: CTGWRank
#Input: SingularValue list, dimensions of Vbig and epsilon
#Output: Rank from CTGW paper
CTGWRank:=proc(Sigma,epsilon,n,m,mu,dimV,d)
local Delta,k,i;
Delta:=evalf(sqrt(n+m));
k:=0;

#find largest gap by 2.4-3 in CGTW
for i from 1 to RowDimension(Sigma)-1 do 
    if Sigma(i) >Delta*epsilon and Sigma(i+1) < epsilon then
	k:=max(i,k);
    fi
od;

if(Sigma(RowDimension(Sigma)) >= epsilon*Delta) then k :=n fi; #full rank case.

if(k=0) then
WARNING("SVD Detected rank of 0. Unable to compute GCRD");
  return fail
fi;  #something went wrong here, no gaps detected. Failed.


#now k should be the approximate numeric rank of Vbig, we now need to scale this 
#appropriately to get some information about the Differential VDM matrix
print(k);
return floor(k*dimV/m); 

end proc;
#############################################################
#############################################################
#############################################################
#Proc: ScaledCTGWRank
#Input: SingularValue list, dimensions of Vbig, n,d,mu and epsilon
#Output: Rank from CTGW paper scaled by the block size

ScaledCTGWRank:=proc(Sigma,epsilon,n,m,mu,dimV,d)
local Delta,k,i;
Delta:=evalf(sqrt(n+m)/(d+1+mu)); #scale by the column block size
k:=0;

#find largest gap by 2.4-3 in CGTW
for i from 1 to RowDimension(Sigma)-1 do 
    if Sigma(i) > Delta*epsilon and  Sigma(i+1) < epsilon then
	k:=max(i,k);
    fi
od;

if(Sigma(RowDimension(Sigma)) >= evalf(epsilon*Delta)) then k := n fi; #full rank case.

if(k=0) then
WARNING("SVD Detected rank of 0. Unable to compute GCRD");
  return fail
fi;  #something went wrong here, no gaps detected. Failed.


#now k should be the approximate numeric rank of Vbig, we now need to scale this 
#appropriately to get some information about the Differential VDM matrix
#print(k);
return ceil(k*dimV/m); #take the ceiling, this ensures that we ignore spurious singular values.

end proc;

#############################################################
#############################################################
#############################################################
#Proc: RatioRank
#Input: SingularValue list, dimensions of Vbig, n  and epsilon
#Output: Rank based on ratios of SingularValues. Looks for the last "big change" in Sigma(i)/Sigma(i+1)
#Andrew gave me this idea
#The idea is to look for a "gap" in the singular values by looking at their ratios. The last "jump" is the declared rank

RatioRank := proc(Sigma,epsilon,n,dimV)

local i,k;
k:=0;
if(Sigma[1] <> 0) then k :=1 fi;
for i from 1 to RowDimension(Sigma) -1 do
    if Sigma[i+1]/Sigma[i] < epsilon* evalf(sqrt(n)) # jump detected
	then k:=i;
    fi;
od;
if(Sigma[RowDimension(Sigma)] > epsilon * evalf(sqrt(n))) then k := RowDimension(Sigma)  fi; #full rank check
return floor(k/n*dimV); #scale rank accordingly
end proc;

#############################################################
#############################################################
#############################################################
#Proc: EpsilonRank
#Input:  epsilon
#Output: Rank based on an epsilon guess
#This is likely to be bad for large systems.

EpsilonRank := proc(Sigma,epsilon,n,dimV)
local i,k;
k:=0;
for i from 1 to RowDimension(Sigma) do 
    if (Sigma[i] > epsilon) then k:=k+1  fi;
od;
print(k);
return floor(k*dimV/n);
end proc;
#############################################################
#############################################################
#############################################################

#Proc: DenomDegreeVector
#Input: Differential polynomial f over R
#Output: RowVector with the degrees of denom( f0,f1,f2...fn)
#
DenomDegreeVector:=proc(f,R)

local fvec;
  fvec:=CoefficientVector(f,R[1]);


  return Vector(degree(f,R[1])+1,i->degree(denom(fvec[i]),R[2]));
end proc;

#############################################################
#############################################################
#############################################################

#Proc: NumerDegreeVector
#Input: Differential polynomial f over R
#Output: RowVector with the degrees of numer( f0,f1,f2...fn)
#
NumerDegreeVector:=proc(f,R)

local fvec;
  fvec:=CoefficientVector(f,R[1]);


  return Vector(degree(f,R[1])+1,i->degree(numer(fvec[i]),R[2]));
end proc;

#############################################################
#############################################################
#############################################################

#Proc: DegreeVector
#Input: Differential polynomial f over R
#Output: RowVector with the degrees of f0,f1,f2...fn
#
DegreeVector:=proc(f,R)

local fvec;
fvec:=CoefficientVector(f,R[1]);


return Vector(degree(f,R[1])+1,i->degree(fvec[i],R[2]));
end proc;

#############################################################
#############################################################
#############################################################

#Proc: CombinedCoefficientVector
#Input: Differential polynomial f over R
#Output: Combined coefficient vector of f
#
CombinedCoefficientVector:=proc(f,R)

local fvec,L;
fvec:=CoefficientVector(f,R[1]);

L:=[];

L:=[seq(convert(CoefficientVector(fvec[i],R[2]),list),i=1..ArrayNumElems(fvec))];

return convert(L,Vector);

end proc;


#############################################################
#############################################################
#############################################################
#Proc DataTrials
#Input: degree bounds on D, d , f/g, amount of noise, SVD neighborhood over Differential Algebra R
#Output: Statistics!
#just use file redirection to capture the output and stuff.

DataTrials:=proc ( bndDD, bndDd, bndF,delta,epsilon,trials,R)
local x, Dx, degX,degG,degF,degreeF,degreeG,degreeH, PerturbationsF,PerturbationsFAvg, PerturbationsG, PerturbationsGAvg,i,failures;
local degH,f,g,h,U,fhat,ghat,fhatWeighted,ghatWeighted,myMax,myMin,avg,stddev,Q25,Q50,Q75,RankFailures;

x:= R[2];
Dx:=R[1];
degX:= rand(bndDd[1]..bndDd[2]);
degH:= rand(bndDD[1]..bndDD[2]);
degF:= rand(bndF[1]..bndF[2]);

RankFailures:=0;
failures:=0;

PerturbationsF:=[];
PerturbationsG:=[]; #store the results

PerturbationsFAvg:=[];
PerturbationsGAvg:=[]; #store the results

for i from 1 to trials do
    try
	degreeF:=degF();
	degreeG:=degF();#not a typo
	degreeH:=degH();
	
	f:=RandDPoly(degX(),min(degreeF,degreeG),R);
	g:=RandDPoly(degX(),max(degreeF,degreeG),R); #lower degree entry goes first.
	h:=RandDPoly(degX(),degreeH,R);
	
	#give a GCRD
	f:=mult(f,h,R);
	g:=mult(g,h,R);
	
	#noise coeffs
	f:=evalf(f+Noise(f,delta,R));
	g:=evalf(g+Noise(g,delta,R));
	
	#compute the GCRD and perturbations
	U:=UnstructuredApproxGCRD(f,g,epsilon,R);
	fhat:=U[2];
	ghat:=U[3];
	
	fhatWeighted:=U[4];
	ghatWeighted:=U[5];
	
	if (degree(U[1],R[1]) > 0) then
	      PerturbationsF:= [op(PerturbationsF),OreNorm(f-fhat,R,2)];
	      PerturbationsG:=[op(PerturbationsG),OreNorm(g-ghat,R,2)];
	  
	      PerturbationsFAvg:= [op(PerturbationsFAvg),OreNorm(f-fhatWeighted,R,2)];
	      PerturbationsGAvg:=[op(PerturbationsGAvg),OreNorm(g-ghatWeighted,R,2)];
	      	
	      	if OreNorm(f-fhat,R,2)/OreNorm(f,R,2) >= epsilon  or 
		    OreNorm(g-ghat,R,2)/OreNorm(g,R,2) >= epsilon or
		    OreNorm(f-fhatWeighted,R,2)/OreNorm(f,R,2) >= epsilon  or
		    OreNorm(g-ghatWeighted,R,2)/OreNorm(g,R,2) >= epsilon   then
			failures:=failures+1;      
		fi; 
	else
	      RankFailures:=RankFailures+1;
	fi;
	
     
	

      catch:
	RankFailures:=RankFailures+1;
      end try;
od;

printf("FAILURES: %d \t RANK FAILURES: %d \n", failures,RankFailures); 
printf("----STATISTICS-----\nNoise: %f -----Neighborhood: %f ----- Trials:  %d\n",delta,epsilon,trials);

myMax:=max(PerturbationsF);
myMin:=min(PerturbationsF);
avg  :=Mean(PerturbationsF);
stddev:= StandardDeviation(PerturbationsF);

Q25:=evalf(Percentile(PerturbationsF,25));
Q50:=evalf(Percentile(PerturbationsF,50));
Q75:=evalf(Percentile(PerturbationsF,75));

print("f statistics\n");
printf("Max: %f \t Min: %f \t Avg: %f \t StdDev: %f \n",myMax,myMin,avg,stddev);
printf("Q25: %f, Q50: %f, Q75: %f\n \n \n",Q25,Q50,Q75);
######################################################
myMax:=max(PerturbationsG);
myMin:=min(PerturbationsG);
avg  :=Mean(PerturbationsG);
stddev:= StandardDeviation(PerturbationsG);

Q25:=evalf(Percentile(PerturbationsG,25));
Q50:=evalf(Percentile(PerturbationsG,50));
Q75:=evalf(Percentile(PerturbationsG,75));

printf("g statistics\n");
printf("Max: %f \t Min: %f \t Avg: %f \t StdDev: %f \n",myMax,myMin,avg,stddev);
printf("Q25: %f, Q50: %f, Q75: %f\n \n \n",Q25,Q50,Q75);


###########################################

myMax:=max(PerturbationsFAvg);
myMin:=min(PerturbationsFAvg);
avg  :=Mean(PerturbationsFAvg);
stddev:= StandardDeviation(PerturbationsFAvg);

Q25:=evalf(Percentile(PerturbationsFAvg,25));
Q50:=evalf(Percentile(PerturbationsFAvg,50));
Q75:=evalf(Percentile(PerturbationsFAvg,75));

printf("f-Averaged statistics\n");
printf("Max: %f \t Min: %f \t Avg: %f \t StdDev: %f \n",myMax,myMin,avg,stddev);
printf("Q25: %f, Q50: %f, Q75: %f\n \n \n",Q25,Q50,Q75);
###############################################
myMax:=max(PerturbationsGAvg);
myMin:=min(PerturbationsGAvg);
avg  :=Mean(PerturbationsGAvg);
stddev:= StandardDeviation(PerturbationsGAvg);

Q25:=evalf(Percentile(PerturbationsGAvg,25));
Q50:=evalf(Percentile(PerturbationsGAvg,50));
Q75:=evalf(Percentile(PerturbationsGAvg,75));

printf("g-Averaged statistics\n");
printf("Max: %f \t Min: %f \t Avg: %f \t StdDev: %f \n",myMax,myMin,avg,stddev);
printf("Q25: %f, Q50: %f, Q75: %f\n \n \n",Q25,Q50,Q75);
###############################################

#in case we want to do more work.
return [PerturbationsF,PerturbationsG,PerturbationsFAvg,PerturbationsGAvg];
end  proc;

#############################################################
#############################################################
#############################################################
#Proc DataTrialsUNormalized
#Input: degree bounds on D, d , f/g, amount of noise, SVD neighborhood over Differential Algebra R
#Output: Statistics!
#just use file redirection to capture the output and stuff.

DataTrialsUNormalized:=proc ( bndDD, bndDd, bndF,delta,epsilon,trials,R)
local x, Dx, degX,degG,degF,degreeF,degreeG,degreeH, PerturbationsF,PerturbationsFAvg, PerturbationsG, PerturbationsGAvg,i,failures;
local degH,f,g,h,U,fhat,ghat,fhatWeighted,ghatWeighted,myMax,myMin,avg,stddev,Q25,Q50,Q75,RankFailures;

x:= R[2];
Dx:=R[1];
degX:= rand(bndDd[1]..bndDd[2]);
degH:= rand(bndDD[1]..bndDD[2]);
degF:= rand(bndF[1]..bndF[2]);

RankFailures:=0;
failures:=0;

PerturbationsF:=[];
PerturbationsG:=[]; #store the results

PerturbationsFAvg:=[];
PerturbationsGAvg:=[]; #store the results

for i from 1 to trials do
    try
	degreeF:=degF();
	degreeG:=degF();#not a typo
	degreeH:=degH();
	
	f:=RandDPolyUnormalized(degX(),min(degreeF,degreeG),R) + RandDPoly(degX(),min(degreeF,degreeG),R);
	g:=RandDPolyUnormalized(degX(),max(degreeF,degreeG),R) + RandDPoly(degX(),max(degreeF,degreeG),R); #lower degree entry goes first.
	h:=RandDPolyUnormalized(degX(),degreeH,R) + RandDPoly(degX(),degreeH,R);
	
	#give a GCRD
	f:=mult(f,h,R);
	g:=mult(g,h,R);
	
	#noise coeffs
	#normalize f and g so   doesn't get crazy...
	f:=f/OreNorm(f,R,2) + Noise(f,delta,R);
	g:=g/OreNorm(g,R,2) + Noise(g,delta,R);
	
	#compute the GCRD and perturbations
	U:=UnstructuredApproxGCRD(f,g,epsilon,R);
	fhat:=U[2];
	ghat:=U[3];
	
	fhatWeighted:=U[4];
	ghatWeighted:=U[5];
	
	
	if (degree(U[1],R[1]) > 0) then
	      PerturbationsF:= [op(PerturbationsF),OreNorm(f-fhat,R,2)];
	      PerturbationsG:=[op(PerturbationsG),OreNorm(g-ghat,R,2)];
	  
	      PerturbationsFAvg:= [op(PerturbationsFAvg),OreNorm(f-fhatWeighted,R,2)];
	      PerturbationsGAvg:=[op(PerturbationsGAvg),OreNorm(g-ghatWeighted,R,2)];
	      
	      	if OreNorm(f-fhat,R,2)/OreNorm(f,R,2) >= epsilon  or 
		  OreNorm(g-ghat,R,2)/OreNorm(g,R,2) >= epsilon or
		  OreNorm(f-fhatWeighted,R,2)/OreNorm(f,R,2) >= epsilon  or
		  OreNorm(g-ghatWeighted,R,2)/OreNorm(g,R,2) >= epsilon   then
			failures:=failures+1;      
		fi;     
	else
	    RankFailures:=RankFailures+1;
	fi;
	
 

      catch:
	RankFailures:=RankFailures+1;
      end try;
od;

printf("FAILURES: %d \t RANK FAILURES: %d \n", failures,RankFailures); 
printf("----STATISTICS-----\nNoise: %f -----Neighborhood: %f ----- Trials:  %d\n",delta,epsilon,trials);

myMax:=max(PerturbationsF);
myMin:=min(PerturbationsF);
avg  :=Mean(PerturbationsF);
stddev:= StandardDeviation(PerturbationsF);

Q25:=evalf(Percentile(PerturbationsF,25));
Q50:=evalf(Percentile(PerturbationsF,50));
Q75:=evalf(Percentile(PerturbationsF,75));

print("f statistics\n");
printf("Max: %f \t Min: %f \t Avg: %f \t StdDev: %f \n",myMax,myMin,avg,stddev);
printf("Q25: %f, Q50: %f, Q75: %f\n \n \n",Q25,Q50,Q75);
######################################################
myMax:=max(PerturbationsG);
myMin:=min(PerturbationsG);
avg  :=Mean(PerturbationsG);
stddev:= StandardDeviation(PerturbationsG);

Q25:=evalf(Percentile(PerturbationsG,25));
Q50:=evalf(Percentile(PerturbationsG,50));
Q75:=evalf(Percentile(PerturbationsG,75));

printf("g statistics\n");
printf("Max: %f \t Min: %f \t Avg: %f \t StdDev: %f \n",myMax,myMin,avg,stddev);
printf("Q25: %f, Q50: %f, Q75: %f\n \n \n",Q25,Q50,Q75);


###########################################

myMax:=max(PerturbationsFAvg);
myMin:=min(PerturbationsFAvg);
avg  :=Mean(PerturbationsFAvg);
stddev:= StandardDeviation(PerturbationsFAvg);

Q25:=evalf(Percentile(PerturbationsFAvg,25));
Q50:=evalf(Percentile(PerturbationsFAvg,50));
Q75:=evalf(Percentile(PerturbationsFAvg,75));

printf("f-Averaged statistics\n");
printf("Max: %f \t Min: %f \t Avg: %f \t StdDev: %f \n",myMax,myMin,avg,stddev);
printf("Q25: %f, Q50: %f, Q75: %f\n \n \n",Q25,Q50,Q75);
###############################################
myMax:=max(PerturbationsGAvg);
myMin:=min(PerturbationsGAvg);
avg  :=Mean(PerturbationsGAvg);
stddev:= StandardDeviation(PerturbationsGAvg);

Q25:=evalf(Percentile(PerturbationsGAvg,25));
Q50:=evalf(Percentile(PerturbationsGAvg,50));
Q75:=evalf(Percentile(PerturbationsGAvg,75));

printf("g-Averaged statistics\n");
printf("Max: %f \t Min: %f \t Avg: %f \t StdDev: %f \n",myMax,myMin,avg,stddev);
printf("Q25: %f, Q50: %f, Q75: %f\n \n \n",Q25,Q50,Q75);
###############################################

#in case we want to do more work.
return [PerturbationsF,PerturbationsG,PerturbationsFAvg,PerturbationsGAvg];
end  proc;


