###########################################################################
############ NON COMMUTATIVE APPROXIMATE GCD FUNCTIONS ####################
###########################################################################
#Prerequistes for this library
#	: GCDviaOpt.mpl - Commutative Approximate GCD library
#	: approxore_functions.mpl - Non commutative numeric approximate GCRD library.
#	:Various built in maple routines:
#	:DETools
#	:PolynomialTools
#	:Optimization
#	:

#read("GCDviaOpt.mpl"):
#read("approxore_functions.mpl");

#Version History

#0.0 - 	Initial prototype, skeleton, definitions

#1.0 - 	Optimization support for the two norm works. 1/infinity norm doesn't
#	seem to work with Maples optimizer. 

#1.1 -	Added de-vectorization of the solution.

#1.2 -	Added required libraries/packages.

#1.3 -	Added full perturbed GCRD method and some testing code. 

#1.4 -	Implemented procedure for approximate nullspace computation

#1.4.1	Implemneted complete packade.

#2.0 -	Complete re-write  of most procedures since I introduced a bug some where

#2.1 -	Added better support to handle the case of the zero polynomial. 

#2.2 -	Added devectorization code. Prototype should be complete.

#3.0 -	started writing functions for GCRDviaNewton.

#3.1 -	Newton Prototype implemented. 

#3.2 -	Least Sqaure Prototype implemented, with improved heuristic.

#3.2 -	Newton Code Updated - method for actual iteration.

#3.3 - Bug Fixes and implementation of modified Newton iteration

############################################################################
############################################################################
############################################################################
#Proc: ApproxGCRDViaLS
#Input: diff polys f,g over algebra R with D the specified GCRD degree
#	with delta an optiona parameter specifying the degrees in R[2]
#	of the GCRD.
#Output: A least square solution to the Bezout coefficient GCRD formulation
#Purpose: Compute a GCRD
#Comments: This method sets up a system of linear equations from
#	   undetermined coefficients of f^* and seeks the least squares
#	   solution to a linear system over RR[t]. Works only if denom is
#	   known in advance.
ApproxGCRDViaLS:=proc(f,g,R,D,delta:=NULL)
local 	V, #differential Vandermonde matrix
	D_h, #degree of GCRD, obtained earlier w/ SVD
	d_t, #degree of differential Vandermonde in T[2]
	N, # dimension of DifferentialVandermonde
	mu, # 2*N*d_t
	i,
	cfvec,
	h_undetermined_coeff_rep, #variable for of GCRD
	h_vars, #variables for h, we fix lcoeff to be 1
	w_undetermined_coeff_rep, #variables for the Bezout coefficients.
				  #we don't return them, but easily could
	w_vars,			  #vars for Bezout
	vars,			  #all the vars we need
	myeqns,			  #polynomial equations for Least Squre solver
	cfec,			  #temp variable folded into myLS
	myLS,			  #linear system of equations to pass into LS solver
	A,			  #A matrix for LS solver	
	b,			  #b matrix for LS solver
	x,			  #solution from LS solver
	my_gcrd			  #plug in solution to h_undetermined_coeff_rep
	;
	
	#prelim vars
	V := DifferentialVandermonde(f, g, R);
	D_h := D;
	N := RowDimension(V);
	d_t := MatrixDegree(V,R[2]);
	mu:=2*N*d_t;
	
	#set up the nsv
	w_vars := [seq(seq('w'[i][j], j = 0 .. mu), i = 1 .. RowDimension(V))];
	w_undetermined_coeff_rep := Vector[row](N);
	for i to N do
	    w_undetermined_coeff_rep[i] := add('w'[i][j-1]*R[2]^(j-1), j = 1 .. mu+1);
	end do:
	
	#setup h
	if(delta <> NULL) then #specified form of GCRD
	    h_vars:=[seq(seq('hh'[i][j-1], j = 1 .. delta[i]+1), i = 1 .. D_h+1)];
	    h_undetermined_coeff_rep := Vector[row](N);
	    for i to D_h+1 do
		#note the extra space we need for product
		h_undetermined_coeff_rep[i] := add('hh'[i][j-1]*R[2]^(j-1), j = 1 .. delta[i]+1);
	    end do:
	else: #initial degrees not specified
	    h_vars:=[seq(seq('hh'[i][j-1], j = 1 .. mu+d_t+1), i = 1 .. D_h+1)];
	    h_undetermined_coeff_rep := Vector[row](N);
	    for i to D_h+1 do
		#note the extra space we need for product
		h_undetermined_coeff_rep[i] := add('hh'[i][j-1]*R[2]^(j-1), j = 1 .. mu+d_t+1);
	    end do:
	fi;
	
	#this makes the undetermined coefficient rep have monic leading coefficient.
	#Obviously this results in a high degree form, but we can prune it down by
	#speciying something lower in delta and re-applying the algorithm.
	h_undetermined_coeff_rep := subs(lcoeff(h_undetermined_coeff_rep[D_h+1], R[2]) = 1, h_undetermined_coeff_rep);
	
	#merge list of vars
	vars := [op(w_vars), op(h_vars)];
	
	#generate polynomial equations
	myeqns := Equate(simplify(w_undetermined_coeff_rep.V-h_undetermined_coeff_rep), Vector[row](N));
	
	#generate linear equations
	myLS:=[];
	for i to nops(myeqns) do
	    cfvec := CoefficientVector(lhs(myeqns[i]), R[2]);
	    myLS := [op(myLS), op(Equate(cfvec, Vector(ArrayNumElems(cfvec))))] 
	end do:
	
	#solve LS problem and return the GCRD.
	A, b := GenerateMatrix(myLS, vars);
	x := LeastSquares(A, b, free = 'param');
	x := subs([seq('param'[i] = 0, i = 1 .. max(RowDimension(A), ColumnDimension(A)))], x);
	my_gcrd := subs(Equate(Vector(vars), x), h_undetermined_coeff_rep);
	
	return FromCoefficientVector(my_gcrd,R[1]);
end proc:

############################################################################
############################################################################
############################################################################


############################################################################
############################################################################
############################################################################
#Proc: BasisOrePoly
#Input: Degree vector degVec and differential Algebra R and the symbol we
#	want to use for our variables.
#Output: Differential polynomial with undetermined coefficients perscribed
#	 to the degree vector with coefficients coming from symbol.

BasisOrePoly := proc(degVec,R,symbol)

  local myPoly,dims,vec;
  
  #handle the case where we have a row or column vector.
  dims:=max(RowDimension(degVec),ColumnDimension(degVec));
  vec:=Vector(dims, i -> BasisPoly(degVec[i],symbol[i-1],R[2]));
  
  return FromCoefficientVector(vec,R[1]);
 
end proc;
############################################################################
############################################################################
############################################################################
#Proc: UndeterminedModifiedCoefRep
#Input: Poly f over Diff algebra R w/ specified symbol
#	satisfying the denominator constraint 
#	(denom is powers of leading coefficient)
#Output: Polynomial ff in undetermined coefficients of f, with list of coeffs.
#Purpose: Feed this into Newton Iteration to get better answers.
#Comments: Replaces limitations with degree vector BasisOrePoly
#	  for Newton based GCRD computation. 
#	  This method will return poor answers if the input is not 
#	  in the required format.
UndeterminedModifiedCoefRep:= proc (f,h_d,R, symbol)
local d, #degree of poly in R[1] (Dt)
      df, #degree of components of f in R[2] (t)
      fvec, #vector representation of f
      i,
      undetermined_coeff_rep,
      cfs, #coefficient list
      init_pt, #initial point,
      numer_vec, #vector to store numerator
      denom_vec #vector to store denominators
      ;
      
d:=degree(f,R[1]);
fvec:=CoefficientVector(f,R[1]);

#get the numerator, accounting for cancellation.
#we are implicitly assumping that the leading coefficient of h is irriducible
#in this implementation because dealing with cancellation is "hard".
numer_vec :=Vector(i->numer(fvec[i]),d+1); #this is actually bad

#detect invalid input format
for i from  1 to d+1 do 
    #this means that our degree structure is not appropriate
    if( degree(denom(fvec[i]),R[2]) <> degree(h_d^(i-1),R[2]) and degree(h_d,R[2]) >0) then
      WARNING("Co-factor has incorrect form. Expect catatstrophic Failure");
      #return FAIL; #better to fail sooner than later.
    fi;
od:

undetermined_coeff_rep:=Vector(d+1); #allocate space for undetermined coefficients
cfs:=[];
init_pt := [];
for i from 1 to d+1 do 
    df:=degree(numer_vec[i],R[2]);
    undetermined_coeff_rep[i]:= add(symbol[i][j-1]*R[2]^(j-1),j=1..df+1);
    cfs:=[op(cfs),seq(symbol[i][j-1],j=1..df+1)];
    init_pt:=[op(init_pt),seq(symbol[i][j-1]= CoefficientVector(numer_vec[i],R[2])[j],j=1..df+1)];
od:
 #* h_d^((d+1-i+1-degree(denom(fvec[i]),R[2]))/degree(h_d,R[2]))
#check for off by one error
#this is multiples that would arise from clearing denominators
denom_vec:=Vector(i->h_d^(i-1 - (degree(denom(fvec[i]),R[2])-degree(h_d^(d+1-i+1),R[2]))/degree(h_d,R[2])) ,d+1);

undetermined_coeff_rep:=Zip(`*`,undetermined_coeff_rep,denom_vec);
#return the coefficient vars list for efficient use of subs later
return [FromCoefficientVector(undetermined_coeff_rep,R[1]),cfs, init_pt];

end proc:
############################################################################
############################################################################
############################################################################
#Proc: UndeterminedCoefRep
#Input: Poly f over Diff algebra R w/ specified symbol
#Output: Polynomial ff in undetermined coefficients of f, with list of coeffs.
#Purpose: Feed this into Newton Iteration to get better answers.
#Comments: Replaces limitations with degree vector BasisOrePoly
#	  for Newton based GCRD computation. 

UndeterminedCoefRep:= proc (f,R, symbol)
local d, #degree of poly in R[1] (Dt)
      df, #degree of components of f in R[2] (t)
      fvec, #vector representation of f
      i,
      undetermined_coeff_rep,
      cfs, #coefficient list
      init_pt #initial point
      ;
      
d:=degree(f,R[1]);
fvec:=CoefficientVector(f,R[1]);

undetermined_coeff_rep:=Vector(d+1); #allocate space for undetermined coefficients

cfs:=[];
init_pt := [];
for i from 1 to ArrayNumElems(fvec) do 
    df:=degree(fvec[i],R[2]);
    undetermined_coeff_rep[i]:= add(symbol[i][j-1]*R[2]^(j-1),j=1..df+1);
    cfs:=[op(cfs),seq(symbol[i][j-1],j=1..df+1)];
    init_pt:=[op(init_pt),seq(symbol[i][j-1]= CoefficientVector(fvec[i],R[2])[j],j=1..df+1)];
od:

#return the coefficient vars list for efficient use of subs later
return [FromCoefficientVector(undetermined_coeff_rep,R[1]),cfs, init_pt];

end proc:

############################################################################
############################################################################
############################################################################
#Proc: UndeterminedRationalCoefRep
#Input: Poly f over Diff algebra R w/ specified symbol
#Output: Polynomial ff in undetermined coefficients of f, with list of coeffs.
#Purpose: Feed this into Newton Iteration to get better answers.
#Comments: Replaces limitations with degree vector BasisOrePoly
#	  for Newton based GCRD computation. 

UndeterminedRationalCoefRep:= proc (f,R, symbol)
local d, #degree of poly in R[1] (Dt)
      dfnumer, #degree of numerator of f in R[2] (t)
      dfdenom,#degree of denom of f in R[2]
      fvec, #vector representation of f
      i,
      undetermined_coeff_rep,
      cfs, #coefficient list
      init_pt #initial point
      ;
      
d:=degree(f,R[1]);
fvec:=CoefficientVector(f,R[1]);

undetermined_coeff_rep:=Vector(d+1); #allocate space for undetermined coefficients

cfs:=[];
init_pt := [];
for i from 1 to ArrayNumElems(fvec) do 
    dfnumer:=degree(numer(fvec[i]),R[2]);
    dfdenom :=degree(denom(fvec[i]),R[2]);
    undetermined_coeff_rep[i]:= add(symbol[i][j-1]*R[2]^(j-1),j=1..dfnumer+1)/
				add(symbol[-i][j-1]*R[2]^(j-1),j=1..dfdenom+1);
    cfs:=[op(cfs),seq(symbol[i][j-1],j=1..dfnumer+1)]; #append numer cfs
    cfs:=[op(cfs),seq(symbol[-i][j-1],j=1..dfdenom+1)]; #append denom cfs
    init_pt:=[op(init_pt),seq(symbol[i][j-1]= CoefficientVector(numer(fvec[i]),R[2])[j],j=1..dfnumer+1)];
    init_pt:=[op(init_pt),seq(symbol[-i][j-1]= CoefficientVector(denom(fvec[i]),R[2])[j],j=1..dfdenom+1)];
od:

#return the coefficient vars list for efficient use of subs later
return [FromCoefficientVector(undetermined_coeff_rep,R[1]),cfs, init_pt];

end proc:
############################################################################
############################################################################
############################################################################
#Proc: OreNormOpt
#Input: Ore Polynomial f with differential algebra R with specified p-norm.
#Output: P-norm^p of f, omitting absolute value if possible.
#Purpose: Compute an OreNorm without using absolute value to enable
#	  a differential optimizer to work.

OreNormOpt:= proc(f,R,p)
  local vec, answerVec;
  if p <> 2 then return fail fi;
  #handle the case of 2-norm for now.
  
  vec:=CoefficientVector(f,R[1]);
  answerVec:=Vector(RowDimension(vec),i-> sqrt(NormOpt(vec[i],R[2],p)));
  return sqrt(add(answerVec[i]^2,i=1..RowDimension(answerVec)));
end proc:

############################################################################
############################################################################
############################################################################


#Proc: NormOpt
#Input: polynomial f, in indeterminant t with specified p-norm.
#output: p-norm of f adjusted for optimization problem.
#Purpose: avoids using the absolute value if possible.
#comments: Input is assumped to be real
NormOpt:= proc(f,t,p)
local myNorm,coefVec;
coefVec:=CoefficientVector(f,t);
    if p = 2 then 
      myNorm:=add((coefVec[i])^2,i=1..RowDimension(coefVec));
    fi;
    
    if p = 1 then 
      myNorm:=add(abs(coefVec[i]),i=1..RowDimension(coefVec));
    fi;
    
    if p = infinity then 
      myNorm:= max(abs(coefVec));
    fi;
    myNorm;
end proc;
############################################################################
############################################################################
############################################################################


#Proc: ApproximateNullspace
#Input: matrix S over \RR[x], variable for the NSV  w, a  p-norm value p.
#output: Solution to the optimization  [ norm{w}=1 AND min[norm{x.S}], where
#	we use our distributed norm.
#Purpose: ApproximateNullspace is used to generate an initial guess for GCRDviaOpt,
#	 as we sometimes have failure without a nullspace vector. 
#comments: this can fail sometimes, and when this fails we're pretty much messed.
#assumptions:
#		S is a differential Sylvester matrix whose input f and g were
#		normalized.
ApproximateNullspace:=proc(S,R,W,p:=2)
##WARNING: p=2 currently...
local w,N,d,mu,constraints,eqns,my_norm,nsv_vars,the_result,my_soln,w_approx;

    if  p <> 2 then return FAIL fi;

    N:=RowDimension(S); #square matrix so it doesn't matter.
    #N=n+m... don't mess this up in your next paper again.
    d:=MatrixDegree(collect(S,R[1]),R[2]);
    mu:=2*N*d;  #easier to specify mu this way.
    nsv_vars:=[seq(seq(W[j][i], i = 0 .. mu), j = 1 .. N)]; #generate the list for later
    w:= Vector[row](N,i->BasisPoly(mu,W[i],R[2])); #these generate the same variables...
    eqns:= collect(w.S,R[2]); #
			  #collect b/c degree can FUBAR which is
			  #SNAQRFU.
			  
    #distributed norm hack. clean it up later.
    my_norm:= OreNormOpt( FromCoefficientVector(eqns,R[1]),R,p)^p;

    constraints:=[sum(nsv_vars[ix]^p,ix=1..nops(nsv_vars))=1];

    the_result:= Minimize(my_norm,constraints);

    my_soln:=GenerateMatrix(the_result[2],nsv_vars)[2];

    w_approx := Vector[row](N,i->FromCoefficientVector(my_soln[(mu+1)*i-mu..(mu+1)*i], R[2]));
    ## add code to check stuff here if needed ##

    return [the_result[2],w_approx];
end proc;
############################################################################
############################################################################
############################################################################




#Proc: DifferentialGCRDviaOpt
#Input: Ore polynomials f,g over differential Algebra R[Dt,t]
#output: p-norm of f adjusted for optimization problem.
#Purpose: avoids using the absolute value if possible.
#comments:
#	1.) Input is assumed to be real.
#	2.) modified 2-norm is used for all norms in this version.
#	3.) it is assumed that  deg_D(f) <= deg_D(g), otherwise we just swap them.
#	    This was done because my Differential Vandermonde matrix works this way... (it is a "feature").
#	4.) W is used as the variable for the nullspace vector, U and V are used for placeholder variables
#	    for $f$ and $g$ respectively. This might conflict a bit with other versions of the code.
#	5.) Currently all the constraints used are:
#		norm(f) = norm(g) = 1, or they are normalized.
#		norm(U) = norm(V) = 1 *
#		W.DifferentialVandermonde(U,V,R) = 0 ** 
#		Norm(W) =1 ***
#		Minimize(norm(f-U) + norm(g-V)).
#		*, ** and *** are to prevent trivial or unintersting solutions.
#TODO:
#	- Verify that the solution we get is infact the one of maximum degree. IE
#	 f = Dt and g = Dt+epsilon then U = Dt+ k*epsilon and V = Dt + (1-k)*epsilon
#	for 0< k <1 is a family of solutions that satisfy our optimization problem.
#	Verify  monic and doesn't have content problems.
#	handle the lack of uniqueness from the solutions.
#	Have a cold beer.

DifferentialGCRDViaOpt := proc(f,g,epsilon,R)

#bunch of variables for stuff and things.
#not really practical to distribute the functions.
local p,ff,gg,f_degree_vector,g_degree_vector,U,V,
      Uvector, Vvector, S, N, d, mu, w, Eqs, EQList,
      objective_function, non_trivial_constraints,
      constraints, initial_guess, f_init,g_init,
      f_init_vector,g_init_vector,initial_pt,
      nsv_initial, i, EQs, the_result, u_vars, v_vars,
      w_vars, vars, Equations, w_data, u_data, v_data,
      v_solution_vector, u_solution_vector, ftil, gtil,
      wtil;
      
    
    #2-norm by default.
    p := 2;
    
    #normalize and prevent recursive problems
    ff := evalf(f/OreNorm(f, R, p));
    gg := evalf(g/OreNorm(g, R, p));
    f_degree_vector := DegreeVector(ff, R);
    g_degree_vector := DegreeVector(gg, R);
    
    for i from 1 to ArrayNumElems(f_degree_vector) do 
      f_degree_vector[i]:=max(f_degree_vector[i],0); #just treat 0 as a variable right now to prevent problems.
    od;
    for i from 1 to ArrayNumElems(g_degree_vector) do 
      g_degree_vector[i]:=max(g_degree_vector[i],0); #just treat 0 as a variable right now to prevent problems.
    od;
    
    
    
    #set up differential sylvester matrix
    U := BasisOrePoly(f_degree_vector, R, 'u');
    V := BasisOrePoly(g_degree_vector, R, 'v');
    Uvector := CoefficientVector(U, R[1]);
    Vvector := CoefficientVector(V, R[1]);
    S := DifferentialVandermonde(U, V, R);
    N := degree(ff, R[1])+degree(gg, R[1]);
    d := max(degree(ff, R[2]), degree(gg, R[2]));
    mu := 2*N*d;
    
    
    #set up the equations and constraints
    w := Vector[row](N, proc (i) options operator, arrow; BasisPoly(mu, 'W'[i], R[2]) end proc);
    EQs := convert(collect(w.S, R[2]), list);
    EQList := [seq(op(Equate(CoefficientVector(EQs[i], R[2]), Vector(RowDimension(CoefficientVector(EQs[i], R[2]))))), i = 1 .. nops(EQs))];
    objective_function := OreNormOpt(U-ff, R, 2)^2+OreNormOpt(V-gg, R, 2)^2;
    #bit of a hack to compute the distributed norm
    non_trivial_constraints := [OreNormOpt(FromCoefficientVector(w, R[1]), R, 2)^2 = 1, OreNormOpt(U, R, 2)^2 = 1, OreNormOpt(V, R, 2)^2 = 1];
    constraints := [op(non_trivial_constraints), op(EQList)];
    
    #use approximateGCRD to get an initial guess for f/g
    #then use approximateNullspace to get a guess for an NSV of
    #f and g.
    initial_guess := UnstructuredApproxGCRD(ff, gg, epsilon, R);
    f_init := initial_guess[2]/OreNorm(initial_guess[2], R, p);
    g_init := initial_guess[3]/OreNorm(initial_guess[3], R, p);
    f_init_vector := CoefficientVector(f_init, R[1]);
    g_init_vector := CoefficientVector(g_init, R[1]);
    initial_pt := [];
    initial_pt := [op(initial_pt), seq(op(Equate(CoefficientVector(Uvector[i], R[2]), CoefficientVector(f_init_vector[i], R[2]))), i = 1 .. RowDimension(f_init_vector))];
    initial_pt := [op(initial_pt), seq(op(Equate(CoefficientVector(Vvector[i], R[2]), CoefficientVector(g_init_vector[i], R[2]))), i = 1 .. RowDimension(g_init_vector))];
    nsv_initial := ApproximateNullspace(DifferentialVandermonde(f_init, g_init, R), R, 'W', p)[1];
    initial_pt := [op(initial_pt), op(nsv_initial)];
   
   
    #run the actual optimization code and see what we get.
    the_result:=Minimize(objective_function, constraints, initialpoint = initial_pt)[2];
    #return the_result;
    
    #generate the variables we need to re-construct our solution.
    u_vars := [seq(op(convert(CoefficientVector(Uvector[i], R[2]), list)), i = 1 .. ArrayNumElems(Uvector))];
    v_vars := [seq(op(convert(CoefficientVector(Vvector[i], R[2]), list)), i = 1 .. ArrayNumElems(Vvector))];
    w_vars := [seq(op(convert(CoefficientVector(w[i], R[2]), list)), i = 1 .. ArrayNumElems(w))];
    vars := [op(w_vars), op(u_vars), op(v_vars)];
   
    Equations := GenerateMatrix(the_result, vars)[2];
    

    
    #devectorize stuff here.
    #indexing  should work, but I need to double check this for an off by one error,
    #since I had one earlier.
    w_data := Equations[1 .. nops(w_vars)];
    u_data := Equations[nops(w_vars)+1 .. nops(w_vars)+nops(u_vars)]; 
    v_data := Equations[1+nops(w_vars)+nops(u_vars) .. nops(vars)];
  
    #I tried to do something really clever with seq, but the code was such
    # mess I couldn't understand what it did after I was sober (but it worked).
        
    u_solution_vector := Vector(degree(f, R[1])+1);
    v_solution_vector := Vector(degree(g, R[1])+1);
    
    
    #devectorize f
    for i to ArrayNumElems(f_degree_vector) do
      u_solution_vector[i] := FromCoefficientVector(u_data[1 .. f_degree_vector[i]+1], R[2]);
      u_data := u_data[f_degree_vector[i]+2 .. ArrayNumElems(u_data)];
    end do;
    ftil := FromCoefficientVector(u_solution_vector, R[1]);
    
    #devectorize g
    for i to ArrayNumElems(g_degree_vector) do 
      v_solution_vector[i] := FromCoefficientVector(v_data[1 .. g_degree_vector[i]+1], R[2]);
      v_data := v_data[g_degree_vector[i]+2 .. ArrayNumElems(v_data)];
    end do;
      gtil := FromCoefficientVector(v_solution_vector, R[1]);
      
      #devectorize w. This could be simplified. double check it. 
    wtil := Vector[row](N,i->FromCoefficientVector(w_data[(mu+1)*i-mu..(mu+1)*i],R[2]));
  
  #f and g should have an exact GCRD or one, so let's return their GCRD too.
  return [ftil, gtil, wtil]; 
end proc;


############################################################################
############################################################################
############################################################################
#Proc: NumericRightDivisionViaOpt
#Input: f,h over diff algebra R
#Output: f^* satisfying f~f^* h
#Purpose: fstar is initial input to GCRDviaNewton
#Comments: This division does not use an initial point, so the
#	  division may not produce a good answer.

NumericRightDivisionViaOpt:=proc(f,h,R,sanitized:=false,threshold:=1e-15)

local ff,
      hh,
      df,#degree of $f$ in R[2]=t
      fstar, #what we want to compute
      myvars, #list of variables
      fstarvec,
      fvec,
      fshvec,
      mySystem, #system we use t ogenerate objective_function
      denom_constraints, #constraints on denominator to ensure a unique solution
      objective_function,#objective function for optimization
      distance, #distance from optimization problem. May or may not be useful
      my_soln, #the actual answer we get!
      solvec, #solution vector for cleanup purposes.
      defaultvars,
      i;
      
#Initial values/normalizations
ff:=f/OreNorm(f,R,2);
hh:=h/OreNorm(h,R,2);
df:=degree(ff,R[2]); 

#a little long... sorry
#set up undetermined coefficients and stuff
fstar := add(FromCoefficientVector(add(fs[j][i]*UnitVector(i, df+1), i = 1 .. df+1), R[2])*R[1]^(j-1)/FromCoefficientVector(add(fs[j][-i]*UnitVector(i, df+1), i = 1 .. df+1), R[2]), j = 1 .. m-d+1);
myvars:=[seq(seq(op([fs[j][i], fs[j][-i]]), i = 1 .. df+1), j = 1 .. m-d+1)];
#essentially, denominators are 1 and everything else is 0.
defaultvars := [seq(op([fs[j][1] = 0, fs[j][-1] = 1]), j = 1 .. m-d+1), seq(seq(op([fs[j][i] = 0, fs[j][-i] = 0]), i = 2 .. df+1), j = 1 .. m-d+1)];

#vectorize some stuff for later computations 
fstarvec := CoefficientVector(fstar, R[1]);
fvec:=CoefficientVector(ff,R[1]);
fshvec:=CoefficientVector(mult(fstar,hh,R),R[1]);

#clear denominators so we can take a norm
mySystem := fshvec-fvec;
for i from 1 to nops(mySystem) do
  mySystem[i] := normal(mySystem[i]*denom(mySystem[i]))
od:

#norm of the common denom is 1. prevents trivial solutions and zero divisions
denom_constraints:= [OreNormOpt(expand(product([seq(denom(fstarvec[i]), i = 1 .. ArrayNumElems(fstarvec))][j], j = 1 .. ArrayNumElems(fstarvec))), R, 2)^2 = 1];
objective_function := OreNormOpt(FromCoefficientVector(mySystem, R[1]), R, 2)^2;

#run the actual optimization
distance, my_soln := op(Minimize(objective_function, denom_constraints));


#sanitize the output for the user.
#this involves approximate gcd and thresholding.
if sanitized = true then 
    solvec:=CoefficientVector(subs(optsoln,fstar),R[1]);
    for i from 1 to ArrayNumElems(solvec) do
	solvec[i]:=NumericLowestTerms(solvec[i],R[2],threshold);
    od:
    return FromCoefficientVector(solvec,R[1]);
fi:

return subs(my_soln,fstar)
end proc:


############################################################################
############################################################################
############################################################################
#Proc: NumericRightDivisionViaLS
#Input: f,h over diff algebra R with optional parameter delta being a degree
#	vector for the denominator.
#Output: f^* satisfying f~f^* h
#Purpose: fstar is initial input to GCRDviaNewton
#Comments: This method sets up a system of linear equations from
#	   undetermined coefficients of f^* and seeks the least squares
#	   solution to a linear system over RR[t]. Works only if denom is
#	   known in advance.
#FIXME:	   Denominator isn't computed properly, all the time.
NumericRightDivisionViaLS:=proc(f,h,R,delta:=NULL,sanitized:=false,threshold:=1e-15)

local ff,
      hh,
      df,#degree of $f$ in R[2]=t
      fstar, #what we want to compute
      myvars, #list of variables
      fstarvec,
      fvec,
      fshvec,
      mySystem, #system we use t ogenerate objective_function
      denom_constraints, #constraints on denominator to ensure a unique solution
      objective_function,#objective function for optimization
      distance, #distance from optimization problem. May or may not be useful
      my_soln, #the actual answer we get!
      solvec, #solution vector for cleanup purposes.
      i,
      default_denom_vars, 
      cramer_denom,
      M,
      fvec_reduced,
      fshvec_reduced,
      myeqns,
      myLS,
      cfvec,
      A,
      b,
      x,
      hvec,
      m,
      d,
      numer_delta, #degree vectors for numers/denom
      denom_delta;
      
#Initial values/normalizations
ff:=f/OreNorm(f,R,2);
hh:=h/OreNorm(h,R,2);
df:=(degree(ff,R[2])+degree(h,R[2])); #this is an upper bound, but it is not tight in practice
m:=degree(ff,R[1]);
d:=degree(hh,R[1]);

if delta = NULL then 
    fstar := add(FromCoefficientVector(add(fs[j][i]*UnitVector(i, df+1), i = 1 .. df+1), R[2])*R[1]^(j-1)/FromCoefficientVector(add(fs[j][-i]*UnitVector(i, df+1), i = 1 .. df+1), R[2]), j = 1 .. m-d+1);
    myvars := [seq(seq(op([fs[j][i], fs[j][-i]]), i = 1 .. df+1), j = 1 .. m-d+1)];
    default_denom_vars := [seq(seq(fs[j][-i] = 0, i = 1 .. df+1), j = 1 .. m-d+1)];
elif delta[1] <> NULL and delta[2] <> NULL then
    numer_delta:=delta[1];
    denom_delta:=delta[2];
    fstar := add(FromCoefficientVector(add(fs[j][i]*UnitVector(i, numer_delta[j]+1), i = 1 .. numer_delta[j]+1), R[2])*R[1]^(j-1)/FromCoefficientVector(add(fs[j][-i]*UnitVector(i, denom_delta[j]+1), i = 1 .. denom_delta[j]+1), R[2]), j = 1 .. m-d+1);
    myvars := [seq(seq(fs[j][i], i = 1 .. numer_delta[j]+1), j = 1 .. m-d+1)];
    myvars :=[op(myvars),seq(seq(op([fs[j][-i]]), i = 1 .. denom_delta[j]+1), j = 1 .. m-d+1)];
    default_denom_vars := [seq(seq(fs[j][-i] = 0, i = 1 .. denom_delta[j]+1), j = 1 .. m-d+1)];
fi:
fstarvec := CoefficientVector(fstar, R[1]);
hvec := CoefficientVector(hh, R[1]);
for i to ArrayNumElems(fstarvec) do
  if(delta = NULL) then 
      ##WARNING###
      ###I might have mucked up the degrees some how when I sat up this system.
      ###The +2 is something weird from (m-d)-(i-)+1, and I don't remember
      ###where it came from, but it makes stuff work.
      cramer_denom := op(solve(denom(fstarvec[i]) = hvec[d+1]^min(i-1, df-degree(hvec[d+1], R[2])), myvars));
  else
  ##WARNING###
  ###this shouldn't be mucked up
      #I hope this works.
      #The idea is set the denominator to be the best "power" of h_d.
      #The idea is we avoid factor matching (hard problem) by keeping it "unsimplified".
      cramer_denom := op(solve(denom(fstarvec[i]) = hvec[d+1]^max(0, ceil(denom_delta[i]/degree(hvec[d+1],R[2]))), myvars));
  fi:
  fstarvec[i] := subs(default_denom_vars, subs(cramer_denom, fstarvec[i])) 
end do;
fstar := FromCoefficientVector(fstarvec, R[1]);

M := Transpose(NumericDivisionBasis(ff, hh, R));
fvec_reduced := convert(DeleteRow(CoefficientVector(ff, R[1]), 1 .. d), Vector);
fshvec_reduced := collect(M.fstarvec, R[2]);
myeqns := Equate(fvec_reduced-fshvec_reduced, Vector(m-d+1));
myLS := [];
for i to nops(myeqns) do
  cfvec := CoefficientVector(normal(denom(lhs(myeqns[i]))*lhs(myeqns[i])), t);
  myLS := [op(myLS), op(Equate(cfvec, Vector(ArrayNumElems(cfvec))))] ;
end do;

A, b := GenerateMatrix(subs(default_denom_vars, [op(myLS), op(default_denom_vars)]), myvars);
x := LeastSquares(A, b, free = 'param');
x := subs([seq(param[i] = 0, i = 1 .. max(RowDimension(A), ColumnDimension(A)))], x);
fstar := subs(Equate(Vector(myvars), x), fstar);

#sanitize the output for the user.
#this involves approximate gcd and thresholding.
if sanitized = true then 
    solvec:=CoefficientVector(fstar,R[1]);
    for i from 1 to ArrayNumElems(solvec) do
	solvec[i]:=NumericLowestTerms(solvec[i],R[2],threshold);
    od:
    return FromCoefficientVector(solvec,R[1]);
fi:

return fstar;
end proc:

############################################################################
############################################################################
############################################################################
#Proc: ApproximateGCRDviaNewton
#Input: f,g,h,fstar,gstar over diff algebra R with N iterations
#Output:  Newton iteration minimization of |f-f*h|^2+|g-g*h|
#Purpose: Do the actual newton iteration.
#Comments: Initial points all need to be specified, human examination is required in some
#	   instances.
ApproximateGCRDviaNewton:=proc(f,g,h_guess,fstar_guess,gstar_guess,R,N)

    local fstar,
	  gstar,
	  h,
	  fstarvec,
	  gstarvec,
	  hvec,
	  fsvars, #lists to keep track of variables
	  gsvars, #lists to keep track of variables
	  hvars, #lists to keep track of variables
	  vars, #variables used for newton iteration
	  init_pt, #initial point for Newton Iteration
	  fsinit,
	  gsinit,
	  hinit,
	  x, #rtable storing points for the iteration itself
	  i, #index,
	  m, #deg f
	  n, #deg g
	  d, #deg h
	  denom_fstar, #temp storage for denom of fstar
	  denom_gstar, #temp storage for denom of gstar
	  fs_guess, #we might mutate fstar/gstar guesses
	  gs_guess, 
	  ff, #mutate f and g 
	  gg,
	  hh,
	  myPsi, #function we are optimizing
	  newton_hessian,
	  newton_gradient,
	  perturbation_size; #size of the perturbation of \Delta f and \Delta g

    m:=degree(f,R[1]);
    n:=degree(g,R[1]);
    d:=degree(h,R[1]);

    fsvars:=[];
    gsvars:=[];
    hvars:=[];
    vars:=[];
    init_pt:=[];
    
    fs_guess:=fstar_guess;
    gs_guess:=gstar_guess;
    ff:=f;
    gg:=g;
    x[0]:=0; #initial value

    
    fstar,fsvars,fsinit := op(UndeterminedCoefRep(fs_guess,R,'fs'));
    gstar,gsvars,gsinit := op(UndeterminedCoefRep(gs_guess,R,'gs'));
    h,hvars,hinit	    := op(UndeterminedCoefRep(h_guess,R,'hh'));

    #This has the effect of fixing the leading coefficient to be whatever we tell it to be.
    h:=subs(hvars[nops(hvars)]=lcoeff(lcoeff(h_guess,R[1]),R[2]),h); #substitute the initial point lc for lc
    hvars:= subsop( nops(hvars)=NULL,hvars); #remove the var for lc(lc(h)).
    


    vars:=[op(fsvars),op(gsvars),op(hvars)];
    init_pt:=[op(fsinit),op(gsinit),op(hinit)];

    myPsi:=OreNormOpt(ff-mult(fstar,h,R),R,2)^2 + OreNormOpt(gg-mult(gstar,h,R),R,2)^2;
    newton_hessian:=VectorCalculus:-Hessian(myPsi,vars);
    newton_gradient:=VectorCalculus:-Gradient(myPsi,vars);

    x[0]:=init_pt;
    for i from 1 to N do 
      x[i]:=convert(Equate(vars, LinearSolve(subs(x[i-1], newton_hessian), subs(x[i-1], newton_hessian.Vector(vars)-newton_gradient))), list);
    od:

    return [ subs(x[N],h), subs(x[N],fstar), subs(x[N],gstar), subs(x[N],myPsi), (newton_hessian),myPsi,vars,x[0],x[N] ];
end proc;


############################################################################
############################################################################
############################################################################
#Proc: ApproxNewtonVars
#Input: f,g,h,fstar,gstar over diff algebra R with N iterations
#Output:  Data structure used for Newton iteration minimization of |f-f*h|^2+|g-g*h|
#Purpose: analyzing the structure of the hessian matrix
#Comments: Initial points all need to be specified, human examination is required in some
#	   instances.
ApproxNewtonVars:=proc(f,g,h_guess,fstar_guess,gstar_guess,R)

    local fstar,
	  gstar,
	  h,
	  fstarvec,
	  gstarvec,
	  hvec,
	  fsvars, #lists to keep track of variables
	  gsvars, #lists to keep track of variables
	  hvars, #lists to keep track of variables
	  vars, #variables used for newton iteration
	  init_pt, #initial point for Newton Iteration
	  fsinit,
	  gsinit,
	  hinit,
	  x, #rtable storing points for the iteration itself
	  i, #index,
	  m, #deg f
	  n, #deg g
	  d, #deg h
	  myPsi, #function we are optimizing
	  newton_hessian,
	  newton_gradient,
	  perturbation_size; #size of the perturbation of \Delta f and \Delta g

    m:=degree(f,R[1]);
    n:=degree(g,R[1]);
    d:=degree(h,R[1]);

    fsvars:=[];
    gsvars:=[];
    hvars:=[];
    vars:=[];
    init_pt:=[];

    fstar,fsvars,fsinit := op(UndeterminedCoefRep(fstar_guess,R,'fs'));
    gstar,gsvars,gsinit := op(UndeterminedCoefRep(gstar_guess,R,'gs'));
    h,hvars,hinit	    := op(UndeterminedCoefRep(h_guess,R,'hh'));
   
   
   #This has the effect of fixing the leading coefficient to be whatever we tell it to be.
   h:=subs(hvars[nops(hvars)]=lcoeff(lcoeff(h_guess,R[1]),R[2]),h); #substitute the initial point lc for lc
   hvars:= subsop( nops(hvars)=NULL,hvars); #remove the var for lc(lc(h)).
    

    vars:=[op(fsvars),op(gsvars),op(hvars)];
    init_pt:=[op(fsinit),op(gsinit),op(hinit)];

    myPsi:=OreNormOpt(f-mult(fstar,h,R),R,2)^2 + OreNormOpt(g-mult(gstar,h,R),R,2)^2;
    newton_hessian:=VectorCalculus:-Hessian(myPsi,vars);
    newton_gradient:=VectorCalculus:-Gradient(myPsi,vars);


    return[vars,myPsi,newton_hessian,h,fstar,gstar,init_pt]

end proc;
############################################################################
############################################################################
############################################################################
#Proc: ApproximateModifiedGCRDviaNewton
#Input: f,g,h,fstar,gstar over diff algebra R with N iterations
#Output:  Data structure used for Newton iteration minimization of |f-f*h|^2+|g-g*h|
#Purpose: analyzing the structure of the hessian matrix
#Comments: Initial points all need to be specified, human examination is required in some
#	   instances.
#	   This is the version for rational functions!	   
ApproximateModifiedGCRDviaNewton:=proc(f,g,h_guess,fstar_guess,gstar_guess,R,N,undetermined_denom:=false)
    local fstar,
	  gstar,
	  h,
	  fstarvec,
	  gstarvec,
	  hvec,
	  fsvars, #lists to keep track of variables
	  gsvars, #lists to keep track of variables
	  hvars, #lists to keep track of variables
	  vars, #variables used for newton iteration
	  init_pt, #initial point for Newton Iteration
	  fsinit,
	  gsinit,
	  hinit,
	  x, #rtable storing points for the iteration itself
	  i, #index,
	  m, #deg f
	  n, #deg g
	  d, #deg h
	  denom_fstar, #temp storage for denom of fstar
	  denom_gstar, #temp storage for denom of gstar
	  fs_guess, #we might mutate fstar/gstar guesses
	  gs_guess, 
	  ff, #mutate f and g 
	  gg,
	  hh,
	  opt_vec, #vector form of optimization problem
	  myPsi, #function we are optimizing
	  newton_hessian,
	  newton_gradient,
	  perturbation_size; #size of the perturbation of \Delta f and \Delta g

    m:=degree(f,R[1]);
    n:=degree(g,R[1]);
    d:=degree(h,R[1]);

    fsvars:=[];
    gsvars:=[];
    hvars:=[];
    vars:=[];
    init_pt:=[];
    
    fs_guess:=fstar_guess;
    gs_guess:=gstar_guess;

    
    #set up h so we can use it's leading coefficient
    h,hvars,hinit	    := op(UndeterminedCoefRep(h_guess,R,'hh'));
    #This has the effect of fixing the leading coefficient to be whatever we tell it to be.
   # h:=subs(hvars[nops(hvars)]=lcoeff(lcoeff(h_guess,R[1]),R[2]),h); #substitute the initial point lc for lc
   # hvars:= subsop( nops(hvars)=NULL,hvars); #remove the var for lc(lc(h)).

    ff:=lcoeff(h,R[1])^(m-d)*f;
    gg:=lcoeff(h,R[1])^(n-d)*g;

    fstar,fsvars,fsinit := op(UndeterminedModifiedCoefRep(fs_guess,lcoeff(h,R[1]),R,'fs'));
    gstar,gsvars,gsinit := op(UndeterminedModifiedCoefRep(gs_guess,lcoeff(h,R[1]),R,'gs'));



    vars:=[op(fsvars),op(gsvars),op(hvars)];
    init_pt:=[op(fsinit),op(gsinit),op(hinit)];

    myPsi:=OreNormOpt(ff-mult(fstar,h,R),R,2)^2 + OreNormOpt(gg-mult(gstar,h,R),R,2)^2;
    newton_hessian:=VectorCalculus:-Hessian(myPsi,vars);
    newton_gradient:=VectorCalculus:-Gradient(myPsi,vars);
    x[0]:=init_pt;
    for i from 1 to N do 
      x[i]:=convert(Equate(vars, LinearSolve(subs(x[i-1], newton_hessian), subs(x[i-1], newton_hessian.Vector(vars)-newton_gradient))), list);
    od:

    return [ subs(x[N],h), subs(x[N],fstar), subs(x[N],gstar), subs(x[N],myPsi), (newton_hessian),myPsi,vars,x[0],x[N] ];
end proc;
############################################################################
############################################################################
############################################################################
#Proc: ApproxModifiedNewtonVars
#Input: f,g,h,fstar,gstar over diff algebra R with N iterations
#Output:  Newton iteration minimization of |f-f*h|^2+|g-g*h|
#Purpose: Do the actual newton iteration.
#Comments: Initial points all need to be specified, human examination is required in some
#	   instances.
#	   This is the version for rational functions!	   
#	   We assume that the leading coefficient of h is irriducible
#	   or else we get combinatorial blow up trying to find the right factors.
ApproxModifiedNewtonVars:=proc(f,g,h_guess,fstar_guess,gstar_guess,R,N)
    local fstar,
	  gstar,
	  h,
	  fstarvec,
	  gstarvec,
	  hvec,
	  fsvars, #lists to keep track of variables
	  gsvars, #lists to keep track of variables
	  hvars, #lists to keep track of variables
	  vars, #variables used for newton iteration
	  init_pt, #initial point for Newton Iteration
	  fsinit,
	  gsinit,
	  hinit,
	  x, #rtable storing points for the iteration itself
	  i, #index,
	  m, #deg f
	  n, #deg g
	  d, #deg h
	  denom_fstar, #temp storage for denom of fstar
	  denom_gstar, #temp storage for denom of gstar
	  fs_guess, #we might mutate fstar/gstar guesses
	  gs_guess, 
	  ff, #mutate f and g 
	  gg,
	  hh,
	  myPsi, #function we are optimizing
	  newton_hessian,
	  newton_gradient,
	  perturbation_size; #size of the perturbation of \Delta f and \Delta g

    m:=degree(f,R[1]);
    n:=degree(g,R[1]);
    d:=degree(h,R[1]);

    fsvars:=[];
    gsvars:=[];
    hvars:=[];
    vars:=[];
    init_pt:=[];
    
    fs_guess:=fstar_guess;
    gs_guess:=gstar_guess;

    
    #set up h so we can use it's leading coefficient
    h,hvars,hinit	    := op(UndeterminedCoefRep(h_guess,R,'hh'));
    #This has the effect of fixing the leading coefficient to be whatever we tell it to be.
    h:=subs(hvars[nops(hvars)]=lcoeff(lcoeff(h_guess,R[1]),R[2]),h); #substitute the initial point lc for lc
    hvars:= subsop( nops(hvars)=NULL,hvars); #remove the var for lc(lc(h)).

    fstar,fsvars,fsinit := op(UndeterminedModifiedCoefRep(fs_guess,lcoeff(h,R[1]),R,'fs'));
    gstar,gsvars,gsinit := op(UndeterminedModifiedCoefRep(gs_guess,lcoeff(h,R[1]),R,'gs'));

    
    ff:=lcoeff(h,R[1])^(min(m-d,degree(denom(fstar),R[2])/degree(lcoeff(h,R[1]))))*f;
    gg:=lcoeff(h,R[1])^(min(n-d,degree(denom(gstar),R[2])/degree(lcoeff(h,R[1]))))*g;



    vars:=[op(fsvars),op(gsvars),op(hvars)];
    init_pt:=[op(fsinit),op(gsinit),op(hinit)];

    myPsi:=OreNormOpt(ff-mult(fstar,h,R),R,2)^2 + OreNormOpt(gg-mult(gstar,h,R),R,2)^2;
    newton_hessian:=VectorCalculus:-Hessian(myPsi,vars);
    newton_gradient:=VectorCalculus:-Gradient(myPsi,vars);
    return[vars,myPsi,newton_hessian,h,fstar,gstar,init_pt,ff,gg]
end proc;
############################################################################
############################################################################
############################################################################
#Proc: ApproximateNaiveModifiedGCRDviaNewton
#Input: f,g,h,fstar,gstar over diff algebra R with N iterations
#Output:  Data structure used for Newton iteration minimization of |f-f*h|^2+|g-g*h|
#Purpose: analyzing the structure of the hessian matrix
#Comments: Initial points all need to be specified, human examination is required in some
#	   instances.
#	   This is the version for rational functions!
#	   This method uses a naive undeterined coefficient representation.
#	   The advantage of this method is that it is less error prone to 
#	   structure deviations
ApproximateNaiveModifiedGCRDviaNewton:=proc(f,g,h_guess,fstar_guess,gstar_guess,R,N,undetermined_denom:=false)
    local fstar,
	  gstar,
	  h,
	  fstarvec,
	  gstarvec,
	  hvec,
	  fsvars, #lists to keep track of variables
	  gsvars, #lists to keep track of variables
	  hvars, #lists to keep track of variables
	  vars, #variables used for newton iteration
	  init_pt, #initial point for Newton Iteration
	  fsinit,
	  gsinit,
	  hinit,
	  x, #rtable storing points for the iteration itself
	  i, #index,
	  m, #deg f
	  n, #deg g
	  d, #deg h
	  denom_fstar, #temp storage for denom of fstar
	  denom_gstar, #temp storage for denom of gstar
	  fs_guess, #we might mutate fstar/gstar guesses
	  gs_guess, 
	  ff, #mutate f and g 
	  gg,
	  hh,
	  lc_fstar,
	  lc_gstar,
	  myPsi, #function we are optimizing
	  newton_hessian,
	  newton_gradient,
	  perturbation_size; #size of the perturbation of \Delta f and \Delta g

    m:=degree(f,R[1]);
    n:=degree(g,R[1]);
    d:=degree(h,R[1]);

    fsvars:=[];
    gsvars:=[];
    hvars:=[];
    vars:=[];
    init_pt:=[];
    
    fs_guess:=fstar_guess;
    gs_guess:=gstar_guess;

    
    #set up h so we can use it's leading coefficient
    h,hvars,hinit	    := op(UndeterminedCoefRep(h_guess,R,'hh'));
    #This has the effect of fixing the leading coefficient to be whatever we tell it to be.
    h:=subs(hvars[nops(hvars)]=lcoeff(lcoeff(h_guess,R[1]),R[2]),h); #substitute the initial point lc for lc
    hvars:= subsop( nops(hvars)=NULL,hvars); #remove the var for lc(lc(h)).

    
    

    fstar,fsvars,fsinit := op(UndeterminedRationalCoefRep(fs_guess,R,'fs'));
    gstar,gsvars,gsinit := op(UndeterminedRationalCoefRep(gs_guess,R,'gs'));
    
 
    vars:=[op(fsvars),op(gsvars),op(hvars)];
    init_pt:=[op(fsinit),op(gsinit),op(hinit)];

    #Let's be honest, this is a hack...

    fstarvec:=CoefficientVector(fstar,R[1]);
    gstarvec:=CoefficientVector(gstar,R[1]);

    #now we need to fix the leading coefficients of the denominators
    #to prevent them from vanishing. This leads to an indefinite Hessian
    #and coressponds to division by zero. 
    
    #we just pluck out the leading coefficeint from fstar and gstar.
    
    #fix coefficient for fstar
    for i from 1 to ArrayNumElems(fstarvec) do
	lc_fstar:=lcoeff(denom(fstarvec[i]),R[2]); #assign variable for lcoeff
	fstarvec[i]:=numer(fstarvec[i]) / subs(lc_fstar = subs(init_pt,lcoeff(denom(fstarvec[i]),R[2])),denom(fstarvec[i])); #plug in init pt
	#I don't like set arithmetic, but...
	vars:=convert(convert(vars,set) minus {lc_fstar},list);
    od:

    #fix coefficient for gstar
    for i from 1 to ArrayNumElems(gstarvec) do
	lc_gstar:=lcoeff(denom(gstarvec[i]),R[2]); #assign variable for lcoeff
	gstarvec[i]:=numer(gstarvec[i]) / subs(lc_gstar = subs(init_pt,lcoeff(denom(gstarvec[i]),R[2])),denom(gstarvec[i])); #plug in init pt
	#best way I could come up with to do this...
	vars:=convert(convert(vars,set) minus {lc_gstar},list);
    od:

    #yes, this is totally a hack... so what? This is the NAIVE
    #version of the code.
    ff:=subs(init_pt, simplify(denom(fstar)*f));
    gg:=subs(init_pt,simplify(denom(gstar)*g));
    
    #this could be done many ways... updating fstar/gstar and clearing
    #denominators
    fstar:=FromCoefficientVector(fstarvec,R[1]);
    fstar:=denom(fstar)*fstar;
    gstar:=FromCoefficientVector(gstarvec,R[1]);
    gstar:=denom(gstar)*gstar;
    
    myPsi:=OreNormOpt(ff-mult(fstar,h,R),R,2)^2 + OreNormOpt(gg-mult(gstar,h,R),R,2)^2;
    newton_hessian:=VectorCalculus:-Hessian(myPsi,vars);
    newton_gradient:=VectorCalculus:-Gradient(myPsi,vars);
    x[0]:=init_pt;
    for i from 1 to N do 
      x[i]:=convert(Equate(vars, LinearSolve(subs(x[i-1], newton_hessian), subs(x[i-1], newton_hessian.Vector(vars)-newton_gradient))), list);
    od:

    return [ subs(x[N],h), subs(x[N],fstar), subs(x[N],gstar), subs(x[N],myPsi), (newton_hessian),myPsi,vars,x[0],x[N],ff,gg ];
end proc;